# -*- coding: utf-8 -*-

"""
Ce module rassemble des décorateurs d'utilité générique, réutilisables
"""

from flask import g, request
from flask_httpauth import HTTPBasicAuth

from rest_client import RestClient

from .. import APIS_REGISTRY, LDAP_API
from .errors import unauthorized


auth = HTTPBasicAuth()
auth.auth_error_callback = unauthorized


@auth.verify_password
def verify_password(username_or_token, password=None):
    """Vérifie avec le serveur LDAP si le token d'authentification
    est toujours valide.
    Elle pourrait aussi vérifier la validité du couple (username, password),
    mais on a décidé par design de ne pas l'implémenter pour forcer
    l'utilisation d'un token.

    La fonction vérifie d'abord dans le registre d'APIs (voir __init__.py)
    si le token utilisé n'a pas déjà été vérifié par un service de confiance.
    Si ce n'est pas le cas, alors une vérification est effectuée auprès de
    l'annuaire LDAP et le résultat est stocké dans le registre d'APIs."""

    key = 'token:{}'.format(username_or_token)
    result = APIS_REGISTRY.get(key)
    if not result:  # on demande une vérification de token
        ldap_client = RestClient(LDAP_API)
        response = ldap_client.post(
            '/auth/verify-token',
            auth=(username_or_token, password),
            headers=request.headers
            )
        if response['status'] != 200:  # token invalide
            return False
        else:
            # on récupère les noms d'utilisateurs correspondant à ce token
            user = response['content']['message']
            g.user = user
            APIS_REGISTRY.set(
                key,
                '{username} {method} {url}'.format(
                    username=response['content']['message'],
                    method=request.method,
                    url=request.url
                    )
                )
            APIS_REGISTRY.expire(key, 3600)
            # on force l'enregistrement dans redis à expirer
            # au bout d'une heure
            return True
    else:
        # le token a déjà été vérifié une fois
        user, _ = result.decode('utf-8').split(']')
        g.user = eval(user+']')
        return True


# EOF
