# -*- coding: utf-8 -*-

from datetime import datetime

from flask import request
from werkzeug.utils import secure_filename

from blueflask.lib.errors import bad_request, internal_error
from blueflask.lib.infos import success

from ... import consts, program, listing
from . import api

import pika
import redis


@api.route('/programs', methods=['GET'])
def get_parsed_listing():
    json_data = request.get_json()
    filename = json_data['filename']
    r = redis.Redis(db=2)
    data = eval(r.get(filename))
    return success(message=data, service_code=consts.SERVICE_CODE)


@api.route('/programs', methods=['POST'])
def parse_listing():
    f = request.files['listing']
    try:
        listingfile = listing.Listing(secure_filename(f.filename))
    except ValueError as exc:
        return bad_request(
            message=str(exc),
            service_code=consts.SERVICE_CODE
            )
    else:
        f.save(listingfile.filepath)
    try:
        connection = pika.BlockingConnection(
            pika.ConnectionParameters(
                host='localhost',
                virtual_host='/dvbbox'
                )
            )
    except Exception:
        # exception de connection lancée par pika
        # il faut que rabbitmq-server soit installé
        # et que le vhost /dvbbox existe déjà
        return internal_error(
            message="serveur RabbitMQ injoignable",
            service_code=consts.SERVICE_CODE
            )
    else:
        channel = connection.channel()
        channel.exchange_declare(
            exchange='parse_listing',
            exchange_type='topic'
            )
        # déclaration idempotente: pas grave si l'exchange existe déjà
        channel.basic_publish(
            exchange='parse_listing',
            routing_key=listingfile.service_id,
            body=listingfile.filepath,
            properties=pika.BasicProperties(delivery_mode=2)
            )
        connection.close()
        return success(service_code=consts.SERVICE_CODE)


@api.route('/programs', methods=['PUT'])
def apply_parsed_listing():
    json_data = request.get_json()
    filename = json_data['filename']
    r = redis.Redis(db=2)
    data = eval(r.get(filename))
    data = {
        program['date']: {
            schedule['timestamp']: {'filepath': schedule['filepath']}
            for schedule in program['schedules']
            }
        for program in data['programs']
        }
    try:
        listingfile = listing.Listing(filename)
    except ValueError as exc:
        return bad_request(
            message=str(exc),
            service_code=consts.SERVICE_CODE
            )
    else:
        listingfile.apply(data)
        return success(service_code=consts.SERVICE_CODE)


@api.route('/programs/<string:date>', methods=['GET'])
def get_program(date):
    data = request.get_json()
    if not data:
        services = consts.SERVICES
    else:
        services = data.get('service_id', 85)
        # au cas où un malin envoie un document JSON
        # mais y met tout sauf 'service_id'
        # on renvoie la programmation de la chaine 85
    message = {
        'programs': [],
        'length': len(services)
        }
    for service_id in services:
        p = program.Program(date, service_id)
        infos = p.infos()
        timestamps = sorted(infos)
        message['programs'].append(
            {
                'date': '{}/{}/{}'.format(
                    date[:2],
                    date[2:4],
                    date[4:]
                    ),
                'service_id': service_id,
                'schedule': [
                    {
                        'date': datetime.fromtimestamp(
                            timestamp
                            ).strftime(
                                '%d/%m/%Y'
                                ),
                        'hour': datetime.fromtimestamp(
                            timestamp
                            ).strftime(
                                '%H:%M:%S'
                                ),
                        'filename': infos[timestamp].split(
                            '/')[-1].split(
                                ':')[0].replace(
                                    '.ts', '')
                        }
                    for timestamp in timestamps
                    ]
                }
            )
    return success(message=message, service_code=consts.SERVICE_CODE)


@api.route('/programs/<string:date>/<string:service_id>', methods=['POST'])
def create_program(date, service_id):
    data = request.get_json()
    filenames = data['filenames']
    listing_filename = '{service_id}_{start}_{stop}'.format(
        service_id=service_id,
        start=date,
        stop=date
        )
    listingfile = listing.Listing(listing_filename)
    with open(listingfile.filepath, 'w') as f:
        for filename in filenames:
            f.write(filename+'\n')
    listingfile.parse_and_apply_one_day(date, filenames)
    return success(service_code=consts.SERVICE_CODE)


@api.route('/programs/<string:date>/<string:service_id>', methods=['DELETE'])
def delete_program(date, service_id):
    name = '{}:{}'.format(date, service_id)
    consts.PROGRAMS.delete(name)
    return success(service_code=consts.SERVICE_CODE)


@api.route('/programs/<string:date>/<string:service_id>', methods=['PUT'])
def sync_program_with_master(date, service_id):
    program.Program(date, service_id).sync()
    return success(service_code=consts.SERVICE_CODE)

# EOF
