# -*- coding: utf-8 -*-

"""
Cette librairie contient les différentes routines
permettant au serveur de bien se comporter dans le cluster.
"""

from .consts import ADDRESS, CONFIG, HOSTNAME

import collections
import os
import shlex
import socket
import subprocess

import redis

__all__ = [
    'Server',
    'discover_master_node',
    'set_master_node'
    ]


class Server(object):
    """Cette classe permet de gérer des informations
    comme l'espace disque, la mémoire, certains process.
    """

    def __init__(self, hostname=None):
        self.hostname = hostname or HOSTNAME
        if 'malagasy.com' not in self.hostname:
            self.hostname += '.malagasy.com'
        if not hostname:
            self.address = ADDRESS
        else:
            self.address = socket.gethostbyname_ex(self.hostname)[2][0]
        r = redis.Redis(host=self.address, db=15)
        self.status = r.get('status')
        if self.status:
            self.status = self.status.decode('utf-8')

    def set_master(self):
        r = redis.Redis(host=self.address, db=15)
        r.set('status', 'master')
        self.status = 'master'

    def set_slave(self):
        r = redis.Redis(host=self.address, db=15)
        r.set('status', 'slave')
        self.status = 'slave'

    def get_master(self):
        """Méthode de récupération du serveur master"""
        if self.status == 'master':
            return self
        elif self.status == 'slave':
            peer = None
            for peer in CONFIG['PEERS']:
                server = Server(peer)
                PEER_INFOS = redis.Redis(
                    host=server.address,
                    socket_timeout=5,
                    db=15
                    )
                status = PEER_INFOS.get('status').decode('utf-8')
                if status == 'master':
                    return server
            else:
                return None

    def partition_size(self, path):
        """Renvoie la taille d'une partition sur le serveur"""
        st = os.statvfs(path)
        free = st.f_bavail * st.f_frsize
        total = st.f_blocks * st.f_frsize
        used = (st.f_blocks - st.f_bfree) * st.f_frsize
        disk_usage = collections.namedtuple('usage', 'total used free')
        result = disk_usage(total, used, free)
        return result

    def usage(self, path):
        size = self.partition_size(path)
        return float(size.used)/float(size.total)

    def media(self):
        """Renvoie la liste des noms de fichiers disponibles"""
        if self.hostname.startswith(HOSTNAME):
            cmd = 'ls {}'.format(CONFIG['APP']['media_folder'])
        else:
            cmd = "ssh {} 'ls {}'".format(
                self.hostname,
                CONFIG['APP']['media_folder']
                )
        process = subprocess.Popen(
            shlex.split(cmd),
            stderr=subprocess.PIPE,
            stdout=subprocess.PIPE
            )
        result = process.communicate()[0].decode('utf-8')
        movies = [i for i in result.split('\n') if i.endswith('.ts')]
        return movies


def discover_master_node():
    """Cette fonction permet de découvrir le serveur
    de diffusion principal dans le cluster
    """
    current_server = Server()
    master_server = current_server.get_master()
    if not master_server:
        if current_server.status == 'master':
            return True
        elif current_server.status == 'slave':
            return None
        else:
            return False
    elif current_server.hostname == master_server.hostname:
        return True
    elif current_server.status == 'slave':
        return master_server.__dict__


def set_master_node(hostname=None):
    """Cette fonction permet de marquer un serveur du cluster
    comme étant le serveur de diffusion principal.
    """
    current_server = Server()
    if not hostname:
        if current_server.status == 'master':
            return None
        elif current_server.status == 'slave':
            master_server = current_server.get_master()
            if master_server:
                master_server.set_slave()
            current_server.set_master()
            return True
    else:
        if current_server.status == 'master':
            current_server.set_slave()
        master_server = Server(hostname)
        master_server.set_master()
        return master_server.__dict__


# EOF
