
# !/usr/bin/env python
# encoding: utf-8

"""
Ce module permet de gérer un, ou un ensemble de, programme.
"""


from rest_client import RestClient

from .channel import Channel
from .consts import CONFIG

import collections
import datetime
import time
import xmltodict


__all__ = [
    'Program',
    'ProgramGuide'
    ]


class Program(object):
    """Définition d'une émission TV"""

    def __init__(self, service_id, start, stop, title, desc=None, **kwargs):

        self.channel = Channel(service_id)
        self.start = datetime.datetime.strptime(start, '%d%m%Y %H:%M:%S %z')
        self.stop = datetime.datetime.strptime(stop, '%d%m%Y %H:%M:%S %z')
        self.title = title
        self.description = desc or ''
        for key, value in kwargs.items():
            setattr(self, key, value)

    def __repr__(self):
        msg = (
            '<Program ({channel}) {start} - {stop}: {title}'.format(
                channel=self.channel.name,
                start=self.start.strftime('%d%m%Y %H:%M:%S %z'),
                stop=self.stop.strftime('%d%m%Y %H:%M:%S %z'),
                title=self.title
                )
            )
        return msg

    def tzconv(self, hours_offset=3, minutes_offset=0):
        seconds = (3600 * hours_offset) + (60 * minutes_offset)
        offset = datetime.timedelta(0, seconds)
        tz = datetime.timezone(offset)
        self.start = self.start.astimezone(tz)
        self.stop = self.stop.astimezone(tz)


class ProgramGuide(object):
    """Ensemble d'émissions TV partageant
    les mêmes date et chaine de diffusion
    """

    def __init__(self, service_id, date):
        self.channel = Channel(service_id)
        self.date = datetime.datetime.strptime(date, '%d%m%Y').date()
        self.guide = []

    def __repr__(self):
        return '<ProgramGuide {channel} {date}>'.format(
            channel=self.channel.name,
            date=self.date.strftime('%d%m%Y')
            )

    @property
    def guide_xmltv(self):
        xmltv = {
            'tv': {
                '@xmlns:xalan': 'http://xml.apache.org/xslt',
                'channel': {
                    '@id': self.channel.name,
                    'display-name': {
                        '#text': self.channel.name
                        },
                    },
                'programme': [],
                }
            }
        for program in self.guide:
            if program:
                if isinstance(program.description, str):
                    text = program.description
                else:
                    text = program.description.get('#text', '')
                infos = {
                    '@start': program.start.strftime('%Y%m%d%H%M%S'),
                    '@stop': program.stop.strftime('%Y%m%d%H%M%S'),
                    '@channel': self.channel.name,
                    'title': dict(program.title),
                    'desc': [
                        {
                            '@lang': 'fr'
                            },
                        {
                            '@lang': 'fr',
                            '#text': text
                            }
                        ],
                    'icon': program.icon
                    }
                xmltv['tv']['programme'].append(infos)
        return xmltv

    def get_tmira_data(self):
        """Récupère les programmes depuis le serveur Tmira epg.blueline.mg"""
        date = self.date.strftime('%Y%m%d')
        client = RestClient(CONFIG['URLS']['tmira'])
        infos = client.get(
            '/{service_id}/xmls/{date}.xml'.format(
                service_id=self.channel.service_id,
                date=date
                )
            )
        blob = infos['content']
        xml = xmltodict.parse(blob)
        data = xml['BroadcastData']['ScheduleData']
        service_id = int(data['ChannelPeriod']['ChannelId'])
        events = data['ChannelPeriod'].get('Event', [])

        if type(events) is collections.OrderedDict:
            events = [events]

        for event in events:
            begintime = event['@beginTime'].split('+03:00')[0]
            start = time.mktime(
                time.strptime(
                    begintime,
                    '%Y-%m-%dT%H:%M:%S'
                    )
                )
            tz = datetime.timezone(datetime.timedelta(0, 10800))
            duration = event['@duration']
            stop = start + int(duration)
            start = datetime.datetime.fromtimestamp(start, tz).strftime(
                '%d%m%Y %H:%M:%S %z'
                )
            stop = datetime.datetime.fromtimestamp(stop, tz).strftime(
                '%d%m%Y %H:%M:%S %z'
                )
            title = event['EpgProduction']['EpgText']['Name']
            desc = event['EpgProduction']['EpgText']['Description']
            if not desc:
                desc = ''
            self.guide.append(Program(service_id, start, stop, title, desc))

# EOF
