#!/usr/bin/env python
# encoding: utf-8


from functools import partial
from multiprocessing.pool import Pool

from .consts import CONFIG
from .consts import FUPDB_CREDENTIALS
from .consts import LOGS
from .consts import REST_CLIENT
from . import core

import records


__all__ = [
    'Product'
    ]


class Product(object):
    """Toute politique de bon usage est centrée autour d'un produit.
    Un produit est identifié par son nom, et ses politiques de bon usage"""

    def __init__(self, name):
        self.name = name
        self.policies = {}
        policies_names = CONFIG['POLICIES']
        policy_names = [
            policy_name for policy_name in policies_names
            if policy_name.startswith(self.name)
            ]
        for policy_name in policy_names:
            data_limit, interval = policies_names[policy_name].split(',', 1)
            self.policies[policy_name] = {
                'data_limit': int(data_limit),
                'interval': interval
                }

    def __repr__(self):
        return '<Product {name}>'.format(name=self.name)

    @property
    def customers(self):
        """Récupération de la liste des clients possédant ce produit"""
        from .customer import Customer
        customers = [
            c for c in Customer.all()
            if c.product.name == self.name
            ]
        return customers

    @staticmethod
    def all():
        policy_names = [i for i in CONFIG['POLICIES']]
        products = set(['_'.join(i.split('_')[:-1]) for i in policy_names])
        return [Product(name) for name in products]

    def apply_policies(self, for_real=False):
        """Application des politiques de bon usage à tous les clients
        abonnés à ce produit."""

        f = partial(core.apply_product_policies, for_real)
        with Pool(4) as p:
            response = p.map(f, self.customers)
        customers = {i[0].name: i[1] for i in response if i[1]}
        policies = []
        for policy_name, policy in self.policies.items():
            policy_infos = {
                'data_limit': policy['data_limit'],
                'name': policy_name,
                'interval': policy['interval'],
                'customers': {}
                }
            policy_customers = policy_infos['customers']
            for customer_id, data in customers.items():
                for info in data:
                    if info['name'] == policy_name:
                        policy_customers[customer_id] = {
                            'total': info['total'],
                            'octets_in': info['octets_in'],
                            'octets_out': info['octets_out'],
                            'details': info['details']
                            }
                        if 'fup_in' in info:
                            policy_customers[customer_id]['fup_in'] = True
                        elif 'fup_out' in info:
                            policy_customers[customer_id]['fup_out'] = True
            policies.append(policy_infos)
        return policies

    def update_customers(self):
        """Récupération de la nouvelle liste de clients depuis aiguillier"""
        from .customer import Customer
        response = REST_CLIENT.get(
            '/fup',
            auth=(
                CONFIG['AIGUILLIER']['username'],
                CONFIG['AIGUILLIER']['password']
                ),
            params={'product': self.name}
            )
        customers = response['content'].get('customers', [])
        existing_customers = {
            customer.name: {
                'product': customer.product,
                'refnum': customer.refnum
                }
            for customer in self.customers
            }
        if not customers:
            # on devrait supprimer tous les clients existants
            # parce qu'Aiguillier nous dit que ce produit n'est
            # lié à aucun client;
            # mais pour des raisons de sécurité on ne touche à rien
            pass
        else:
            customers = {
                customer['name']: {
                    'product': customer['product'],
                    'refnum': customer['refnum']
                    }
                for customer in customers
                }
            for customer, infos in customers.items():
                existing_customer = existing_customers.get(customer)
                if not existing_customer:
                    # le client n'existe pas dans la base
                    # on le créé
                    url = core.database_url(**FUPDB_CREDENTIALS)
                    with records.Database(url) as db:
                        sql = (
                            "INSERT INTO customer (name, refnum, product) "
                            "VALUES (:name, :refnum, :product)"
                            )
                        db.query(
                            sql,
                            name=customer,
                            refnum=infos['refnum'],
                            product=infos['product']
                            )
                else:
                    # on vérifie si son refnum a changé
                    if existing_customer['refnum'] != infos['refnum']:
                        # on met à jour le refnum
                        url = core.database_url(**FUPDB_CREDENTIALS)
                        with records.Database(url) as db:
                            sql = (
                                "UPDATE customer SET refnum=:refnum "
                                "WHERE name=:name"
                                )
                            db.query(
                                sql,
                                refnum=infos['refnum'],
                                name=customer
                                )
                        log_msg = (
                            "Changement de refnum de {}: {} => {}"
                            ).format(
                                customer,
                                existing_customer['refnum'],
                                infos['refnum']
                                )
                        LOGS.logger.info(log_msg)
                    # on vérifie si le produit a changé
                    current_product = existing_customer['product']
                    if current_product.name != infos['product']:
                        with records.Database(url) as db:
                            sql = (
                                "UPDATE customer SET product=:product "
                                "WHERE name=:name"
                                )
                            db.query(
                                sql,
                                refnum=infos['product'],
                                name=customer
                                )
                        log_msg = (
                            "Changement de produit de {}: {} => {}"
                            ).format(
                                customer,
                                current_product.name,
                                infos['product']
                                )
                        LOGS.logger.info(log_msg)
                        # on défup de l'ancien produit si nécessaire
                        customer_status = Customer(customer).fup_status
                        for policy_name in current_product.policies:
                            status = customer_status.get(policy_name)
                            if status:
                                status = sorted(
                                    status,
                                    key=lambda x: x['datetime'],
                                    reverse=True
                                    )
                                if status[0]['status']:
                                    # si le client était fup, il faut le défup
                                    REST_CLIENT.delete(
                                        '/fup',
                                        auth=(
                                            CONFIG['AIGUILLIER']['username'],
                                            CONFIG['AIGUILLIER']['password']
                                            ),
                                        params={'customer': self.name}
                                        )
# EOF
