# -*- coding: utf-8 -*-

from datetime import datetime
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from .consts import APIGW
from .consts import CONFIG
from .consts import REDIS_DB

import calendar
import smtplib

import records
import requests


class Product_GP(object):

    def __init__(self, customer_name, msisdn, **kwargs):
        self.msisdn = msisdn[-9:]
        self.customer_name = customer_name
        self.email = kwargs.get('email')
        self.phonenumber = kwargs.get('phonenumber')
        self.notif_mail_percentage = kwargs.get('notif_mail_percentage')
        self.notif_sms_percentage = kwargs.get('notif_sms_percentage')
        self.limit = kwargs.get('limit')
        self.send_sms = kwargs.get('send_sms', 'faux')
        self.username = kwargs.get('username', 'fup-services')

    def __repr__(self):
        return '<Product_GP {msisdn}>'.format(
            name=self.customer_name,
            msisdn=self.msisdn
            )

    @staticmethod
    def all(username='fup-services'):
        bluebase_url = (
            'https://{fqdn}/bluebase'
            '/v1/fup/tech_4G_fup_list'
            ).format(
                fqdn=APIGW
                )
        response = requests.post(
            url=bluebase_url,
            auth=(username, '')
            )
        if response.status_code == 200:
            response = response.json()
            data = response['root']['data']
            infos = data['list']['produit']
            if isinstance(infos, dict):
                infos = [infos]
            result = []
            for info in infos:
                name = '{} {}'.format(
                    info['client_nom'],
                    info['client_prenom']
                    )
                msisdn = info['msisdn']
                kwargs = {
                    'limit': float(info['fup']) * 1024 * 1024 * 1024,
                    'notif_mail_percentage': float(info['notif_mail']),
                    'notif_sms_percentage': float(info.get('notif_phone', 0)),
                    'email': info.get('mail'),
                    'phonenumber': info.get('phone'),
                    'send_sms': info['send_sms'].lower()
                    }
                result.append(Product_GP(name, msisdn, **kwargs))
            return result

    @property
    def consumption(self):
        """Consommation (en Bytes) sur les 30 derniers jours"""
        url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
            eng=CONFIG['CDRDATA']['engine'],
            user=CONFIG['CDRDATA']['username'],
            password=CONFIG['CDRDATA']['password'],
            host=CONFIG['CDRDATA']['host'],
            port=CONFIG['CDRDATA']['port'],
            name=CONFIG['CDRDATA']['name']
            )
        with records.Database(url) as db:
            sql = (
                'SELECT sum(totalflux) FROM '
                'cdr.get_info_traffic_30(:msisdn)'
                )
            rows = db.query(sql, msisdn='261{}'.format(self.msisdn))
            for row in rows:
                value = row[0]
                if not value:
                    value = 0
                else:
                    value = float(value)
                return float(value)

    def get_consumption_percentage(self):
        """Retourne le pourcentage de conso en fonction
        de la limite autorisée"""
        return float((self.consumption / self.limit) * 100)

    def notify(self):
        """Notification"""
        key = 'notification:{msisdn}'.format(msisdn=self.msisdn)
        today = datetime.now()
        last_day_of_month = calendar.monthrange(today.year, today.month)[1]
        remaining_seconds = (last_day_of_month - today.day + 1) * 24 * 3600
        last_notification_email = REDIS_DB.hget(
            '{}:email'.format(key),
            'date'
            )
        last_notification_sms = REDIS_DB.hget(
            '{}:sms'.format(key),
            'date'
            )
        consumption = self.consumption
        percentage = (round(float(consumption)) / self.limit) * 100
        status = False
        if not last_notification_email and not last_notification_sms:
            # aucune notification durant ce mois
            # on fait une notification par email
            if self.notif_mail_percentage:
                if percentage >= self.notif_mail_percentage:
                    if self.send_sms == 'faux':
                        message = (
                            "Cher client, vous vous approchez "
                            "d’une consommation mensuelle équivalente "
                            "au double de la consommation moyenne des "
                            "utilisateurs internet particuliers.\n"
                            "Arrivé à 100 Go d’usage, le débit de votre "
                            "connexion sera dégradé.\n"
                            "Nous vous invitons à consulter vos "
                            "dernières consommations sur le site "
                            "http://espace-client.blueline.mg"
                            )
                    elif self.send_sms == 'vrai':
                        message = (
                            "Cher client, vous avez consommé 90% "
                            "de votre forfait.\n"
                            "Vous pouvez aller gratuitement sur le site "
                            "http://espace-client.blueline.mg "
                            "pour recharger en prépayé."
                            )
                    # envoyer msg
                    content = MIMEText(message, 'plain', 'utf-8')
                    msg = MIMEMultipart()
                    msg['Subject'] = '[blueline] Notification FUP'
                    msg['From'] = CONFIG['ALERT']['from']
                    msg['To'] = self.email
                    msg.attach(content)
                    s = smtplib.SMTP('localhost')
                    s.sendmail(
                        CONFIG['ALERT']['from'],
                        [self.email],
                        msg.as_string()
                        )
                    s.quit()
                    REDIS_DB.hmset(
                        '{}:email'.format(key),
                        {
                            'date': today.strftime('%d%m%Y'),
                            'consumption': consumption,
                            'limit': self.limit
                            }
                        )
                    REDIS_DB.expire(
                        '{}:email'.format(key),
                        remaining_seconds
                        )
                    status = True
            return 'email', percentage, status
        elif last_notification_email and not last_notification_sms:
            # on fait une notification par SMS
            if self.notif_sms_percentage:
                if percentage >= self.notif_sms_percentage:
                    if self.send_sms == 'vrai':
                        # envoyer SMS
                        sms = (
                            "Cher client, nous vous rappellons "
                            "que vous pouvez recharger votre connexion 4G "
                            "en prepaye en allant gratuitement sur le site "
                            "http://espace-client.blueline.mg"
                            )
                        requests.post(
                            url='https://{fqdn}/smsc/v1/sms'.format(
                                fqdn=APIGW
                                ),
                            auth=(self.username, ''),
                            json={
                                'message': sms,
                                'msisdn': self.phonenumber
                                }
                            )
                        REDIS_DB.hmset(
                            '{}:sms'.format(key),
                            {
                                'date': today.strftime('%d%m%Y'),
                                'consumption': consumption,
                                'limit': self.limit
                                }
                            )
                        REDIS_DB.expire(
                            '{}:sms'.format(key),
                            remaining_seconds
                            )
                        status = True
            return 'sms', percentage, status
        else:
            return None

# EOF
