#!/usr/bin/env python
# encoding: utf-8

from datetime import datetime
from email.mime.multipart import MIMEMultipart
from email.mime.base import MIMEBase
from email.mime.text import MIMEText
from jinja2 import Template
from multiprocessing import Pool
from functools import partial

from .. import core
from ..consts import CONFIG
from ..consts import RAPPORT_HTML
from ..consts import REST_CLIENT
from ..consts import FUPDB_CREDENTIALS
from ..customer import Customer
from ..product import Product
from ..product_gp import Product_GP
from . import app

import click
import smtplib
import tablib
import records


@click.group()
def cmd():
    pass


@cmd.command()
@click.option('--host', default='0.0.0.0')
@click.option('--port', default=5000)
def runserver(host, port):
    """Lancer le serveur Werkzeug pour l'API REST (dev. seulement)"""
    app.run(host=host, port=port, debug=True)


@cmd.command()
@click.option('--last-date', default=False)
@click.option('--product-ese', is_flag=True, default=False)
def sync(last_date, product_ese):
    """Mise à jour de la liste des clients Business ainsi que de leurs \
    relevés de consommation superieur a une date."""
    if not product_ese:
        products = Product.all()
        for product in products:
            product.update_customers()
            f = partial(core.update_customer_cdr, last_date)
            with Pool(processes=2) as pool:
                pool.map(f, product.customers)
    else:
        product_eses = Product.all_ese()
        for product in product_eses:
            product.update_customers(product_ese=product_ese)
            f = partial(core.update_customer_cdr, last_date)
            with Pool(processes=2) as pool:
                pool.map(f, product.customers_ese)


@cmd.command()
@click.option('--start-date', default=False)
@click.option('--stop-date', default=False)
def notify(start_date, stop_date):
    """Commande pour notifier les clients grand public \
    selon leur niveau de consommation"""
    click.echo('Récupération des données depuis Bluebase')
    msg_client = ' '
    product_types = CONFIG['PRODUCT']['product_type'].split(',')
    for product_type in product_types:
        customers = Product_GP.all(product_type)
        for customer in customers:
            click.echo('Notification du client {}'.format(customer.ident))
            url = core.database_url(**FUPDB_CREDENTIALS)
            with records.Database(url) as db:
                sql = "SELECT refnum, product FROM customer WHERE name=:name"
                rows = db.query(sql, name=customer.ident.lower())
                row = rows.first()
                if not row and customer.type_ident.lower() == 'ident':
                    msg_client += "Le client {name} n'existe pas.\n".format(
                        name=customer.ident.lower()
                        )
                    click.echo('customer {} exist pas'.format(
                        customer.ident.lower()
                        ))
                else:
                    response = customer.notify(start_date, stop_date)
                    if not response:
                        click.echo("Pas de notification à envoyer")
                    else:
                        medium, percentage, status = response
                        msg = '{}: {}%'.format(medium, round(percentage))
                        if status:
                            msg = '{} ({})'.format(msg, 'Notification envoyée')
                        click.echo(msg)
        if msg_client.strip() != '':
            dest = CONFIG['ALERT']['to_1']
            content = MIMEText(msg_client, 'plain', 'utf-8')
            msg = MIMEMultipart()
            msg['Subject'] = '[blueline] customer inconnu'
            msg['From'] = CONFIG['ALERT']['from']
            msg['To'] = CONFIG['ALERT']['to_1']
            msg.attach(content)
            s = smtplib.SMTP('localhost')
            s.sendmail(
                CONFIG['ALERT']['from'],
                dest.split(','),
                msg.as_string()
                )
            s.quit()


@cmd.command()
@click.option('--start-date', default=False)
@click.option('--stop-date', default=False)
def notify_try(start_date, stop_date):
    """Commande pour notifier les clients grand public \
    selon leur niveau de consommation"""
    click.echo('Récupération des données depuis Bluebase')
    msg_client = ' '
    product_types = CONFIG['PRODUCT_TRY']['product_type'].split(',')
    for product_type in product_types:
        customers = Product_GP.all(product_type)
        for customer in customers:
            click.echo('Notification du client {}'.format(customer.ident))
            url = core.database_url(**FUPDB_CREDENTIALS)
            with records.Database(url) as db:
                sql = "SELECT refnum, product FROM customer WHERE name=:name"
                rows = db.query(sql, name=customer.ident.lower())
                row = rows.first()
                if not row and customer.type_ident.lower() == 'ident':
                    msg_client += "Le client {name} n'existe pas.\n".format(
                        name=customer.ident.lower()
                        )
                    click.echo('customer {} exist pas'.format(
                        customer.ident.lower()
                        ))
                else:
                    response = customer.notify_try(start_date, stop_date)
                    if not response:
                        click.echo("Pas de notification à envoyer")
                    else:
                        medium, percentage, status = response
                        msg = '{}: {}%'.format(medium, round(percentage))
                        if status:
                            msg = '{} ({})'.format(msg, 'Notification envoyée')
                        click.echo(msg)
        if msg_client.strip() != '':
            dest = CONFIG['ALERT']['to_1']
            content = MIMEText(msg_client, 'plain', 'utf-8')
            msg = MIMEMultipart()
            msg['Subject'] = '[blueline] customer inconnu'
            msg['From'] = CONFIG['ALERT']['from']
            msg['To'] = CONFIG['ALERT']['to_1']
            msg.attach(content)
            s = smtplib.SMTP('localhost')
            s.sendmail(
                CONFIG['ALERT']['from'],
                dest.split(','),
                msg.as_string()
                )
            s.quit()


@cmd.command()
@click.option('--product', is_flag=True, default=False)
def reconciled(product):
    if not product:
        """compare les clients fupe dans aiguillier et api-fup"""
        subject = '[blueline] Reconcile FUP'
        dataset_reconcile = tablib.Dataset()
        message = ''
        customers = []
        dataset_reconcile.headers = [
            'customers',
            'fup_aiguillier',
            'fup_api'
            ]

        response = REST_CLIENT.get(
            '/fup',
            auth=(
                CONFIG['AIGUILLIER']['username'],
                CONFIG['AIGUILLIER']['password']
                )
            )
        infos = response['content']['fup']
        infos_aiguillier = [
                                info['name'].lower()
                                for info in infos
                                if info['name']
                                ]
        url = core.database_url(**FUPDB_CREDENTIALS)
        with records.Database(url) as db:
            sql = (
                "SELECT * FROM fup AS data "
                "WHERE datetime = "
                "(SELECT MAX(datetime) FROM fup "
                "WHERE customer = data.customer) "
                "and status = True"
                )
            rows = db.query(sql)
            infos_api = [row.customer for row in rows]
        customers = [x for x in infos_aiguillier if x not in customers]
        for info in infos_api:
            if info not in customers:
                customers.append(info)
        for customer in customers:
            result_aiguillier = 'NO'
            result_api = 'NO'
            if customer in infos_aiguillier:
                result_aiguillier = 'YES'
            if customer in infos_api:
                result_api = 'YES'
            dataset_reconcile.append(
                [
                    customer,
                    result_aiguillier,
                    result_api
                    ]
                )
        message += str(dataset_reconcile)
    else:
        bool_error = False
        dataset_reconcile = tablib.Dataset()
        message = ''
        customers = {}
        dataset_reconcile.headers = [
            'customers',
            'product_aiguillier',
            'refnum_aiguillier',
            'product_bluebase',
            'refnum_bluebase',
            'product_gateway',
            'refnum_gateway'
            ]
        products = Product.all()
        click.echo(products)
        """ customer from aiguillier """
        for product in products:
            customers_api = product.get_customers_aiguillier()
            customers.update(customers_api)
        """ customer from 4d """
        for product in products:
            customers_bluebase = product.get_customers_bluebase()
            for customer, infos in customers_bluebase.items():
                if customer.lower() not in customers.keys():
                    click.echo('customer 4D {}'.format(
                        customer
                        ))
                    customers.update({
                        customer.lower(): {
                            'product': 'unknown',
                            'refnum': 'unknown',
                            'product_bluebase': infos['product'],
                            'refnum_bluebase': infos['refnum']
                            }
                        })
                else:
                    customers[customer.lower()].update({
                        'product_bluebase': infos['product'],
                        'refnum_bluebase': infos['refnum']
                        })
        """ customer from api_gateway"""
        for product in products:
            customers_gateway = product.get_customers_gateway()
            for customer, infos in customers_gateway.items():
                if customer.lower() not in customers.keys():
                    click.echo('customer gateway {}'.format(
                        customer
                        ))
                    customers.update({
                        customer.lower(): {
                            'product': 'unknown',
                            'refnum': 'unknown',
                            'product_gateway': infos['product'],
                            'refnum_gateway': infos['refnum']
                            }
                        })
                else:
                    customers[customer.lower()].update({
                        'product_gateway': infos['product'],
                        'refnum_gateway': infos['refnum']
                        })
        customers = sorted(
            customers.items(),
            key=lambda x: x[1]['product']
            )
        for customer, infos in customers:
            product = infos.get('product', 'unknown')
            refnum = infos.get('refnum', 'unknown')
            refnum_gateway = infos.get('refnum_gateway', 'unknown')
            product_gateway = infos.get('product_gateway', 'unknown')
            refnum_bluebase = infos.get('refnum_bluebase', 'unknown')
            product_bluebase = infos.get('product_bluebase', 'unknown')
            condition = (
                product_gateway != product
                or
                refnum_gateway != refnum
                or
                product_bluebase.lower() != product
                or
                refnum_bluebase != refnum
                )
            if condition:
                bool_error = True
                dataset_reconcile.append(
                    [
                        customer.lower(),
                        product.lower(),
                        refnum,
                        product_bluebase.lower(),
                        refnum_bluebase,
                        product_gateway.lower(),
                        refnum_gateway
                        ]
                    )
        if bool_error:
            subject = '[blueline] Incohérence Reconciliation customer'
            message += str(dataset_reconcile)+"\n"
        else:
            subject = '[blueline] Succes Reconciliation customer'
            message += "Succes"
    with open('/tmp/details_reconciliation.csv', 'w') as f:
        content = dataset_reconcile.export('csv', delimiter=';')
        f.write(content)
    # envoyer msg
    email = CONFIG['ALERT']['to_1']
    content = MIMEText(message, 'plain', 'utf-8')
    attachment = MIMEBase('text', 'csv')
    fp = open('/tmp/details_reconciliation.csv', 'rb')
    attachment.set_payload(fp.read())
    fp.close()
    attachment.add_header(
        'Content-Disposition',
        'attachment',
        filename='details_reconciliation.csv'
        )
    msg = MIMEMultipart('related')
    msg['Subject'] = subject
    msg['From'] = CONFIG['ALERT']['from']
    msg['To'] = email
    msg.attach(content)
    msg.attach(attachment)
    s = smtplib.SMTP('localhost')
    s.sendmail(
        CONFIG['ALERT']['from'],
        email.split(','),
        msg.as_string()
        )
    s.quit()


@cmd.command()
@click.option('--for-real', is_flag=True, default=False)
@click.option('--send-mail', is_flag=True, default=False)
@click.option('--random', is_flag=True, default=False)
def apply(for_real, send_mail, random):
    """Appliquer les politiques de FUP pour tous les clients Business"""

    def str_data(conso_gigabytes):
        result = str(conso_gigabytes).replace('.', ',')
        if result == '0,0':
            return '0'
        else:
            return result

    result = {}
    products = Product.all()
    products = sorted(products, key=lambda x: x.name, reverse=True)
    for product in products:
        policies = product.apply_policies(for_real=for_real, random=random)
        if policies:
            result[product.name] = policies
    if result:
        message = ''
        dataset_statistics = tablib.Dataset()
        dataset_statistics.headers = [
            'product',
            'fup_in',
            'fup_out'
            ]
        dataset_synthesis = tablib.Dataset()
        dataset_synthesis.headers = [
            'policy_name',
            'product',
            'false',
            'true',
            'total',
            'true/total'
            ]
        dataset = tablib.Dataset()
        dataset.headers = [
            'product',
            'data_limit',
            'policy_name',
            'customer',
            'total_download',
            'total_upload',
            'download_national',
            'upload_national',
            'total',
            'status',
            'date'
            ]
        products = sorted([i for i in result], reverse=True)
        for product in products:
            policies = result[product]
            message += 'Produit: {product}\n'.format(product=product)
            number_fup_in = 0
            number_fup_out = 0
            for policy in policies:
                message += (
                    '\n'
                    'Limite autorisée: {data_limit} Go\n'
                    ).format(
                        data_limit=policy['data_limit']
                        )
                foo = ''
                fup_true = 0
                total_client = len(policy['customers'])
                for customer, data in policy['customers'].items():
                    customer_policies_status = Customer(customer).fup_status
                    policy_status = False
                    customer_policy_status = customer_policies_status.get(
                        policy['name']
                        )
                    if customer_policy_status:
                        customer_policy_status = sorted(
                            customer_policy_status,
                            key=lambda x: x['datetime'],
                            reverse=True
                            )
                        policy_status = customer_policy_status[0]['status']
                    conso = {
                        'internet': {'octets_in': 0, 'octets_out': 0},
                        'national': {'octets_in': 0, 'octets_out': 0},
                        'cache': {'octets_in': 0, 'octets_out': 0}
                    }
                    for source in data['sources']:
                        source_name = source['source']
                        octets_in = source['octets_in']
                        octets_out = source['octets_out']
                        conso[source_name] = source
                        conso[source_name]['octets_in'] = str_data(
                            round(octets_in/1000000000, 2)
                            )
                        conso[source_name]['octets_out'] = str_data(
                            round(octets_out/1000000000, 2)
                            )
                    total_octets_in = str_data(data['octets_in'])
                    total_octets_out = str_data(data['octets_out'])
                    total = str_data(data['total'])
                    dataset.append(
                        [
                            product,
                            str(policy['data_limit']).replace('.', ','),
                            policy['name'],
                            customer,
                            total_octets_in,
                            total_octets_out,
                            conso['national']['octets_in'],
                            conso['national']['octets_out'],
                            total,
                            policy_status,
                            datetime.now().date().strftime('%d/%m/%Y')
                            ]
                        )
                    if policy_status is True:
                        fup_true += 1
                    if data.get('fup_in'):
                        number_fup_in = number_fup_in + 1
                        status = 'fup_in'
                    elif data.get('fup_out'):
                        status = 'fup_out'
                        number_fup_out = number_fup_out + 1
                    else:
                        status = None
                    if status:
                        foo += (
                            ' - {customer}: {total} Go ({status})\n'.format(
                                customer=customer,
                                total=data['total'],
                                status=status
                                )
                            )
                if foo:
                    message += foo
                else:
                    message += "Aucun client à FUPer\n"
                if total_client == 0:
                    total_client = 1
                fup_false = total_client - fup_true
                pourcentage_fup = fup_true * 100 / total_client
                pourcentage_fup = round(pourcentage_fup)
                pourcentage_fup = '{}%'.format(pourcentage_fup)
                dataset_synthesis.append(
                    [
                        policy['name'],
                        product,
                        fup_false,
                        fup_true,
                        total_client,
                        pourcentage_fup
                        ]
                    )
            dataset_statistics.append(
                [
                    product,
                    number_fup_in,
                    number_fup_out
                    ]
                )
            message += '\n\n\n'
        message += str(dataset_statistics)
        message += '\n\n\n'
        message += str(dataset_synthesis)
        message_html = Template(RAPPORT_HTML).render(
            date=datetime.now().date().strftime('%d/%m/%Y'),
            result=result,
            stats=dataset_statistics,
            synthesis=dataset_synthesis
            )
        click.echo(message)
        with open('/tmp/details.csv', 'w') as f:
            f.write(dataset.export('csv', delimiter=';'))
        if send_mail:
            recipients = CONFIG['ALERT']['to']
            content_text = MIMEText(message, 'plain', 'utf-8')
            content_html = MIMEText(message_html, 'html', 'utf-8')
            attachment = MIMEBase('text', 'csv')
            fp = open('/tmp/details.csv', 'rb')
            attachment.set_payload(fp.read())
            fp.close()
            attachment.add_header(
                'Content-Disposition',
                'attachment',
                filename='details.csv'
                )

            msg = MIMEMultipart('related')
            msg['Subject'] = CONFIG['ALERT']['subject']
            msg['From'] = CONFIG['ALERT']['from']
            msg['To'] = recipients
            msg_body = MIMEMultipart('alternative')
            msg_body.attach(content_html)
            msg_body.attach(content_text)
            msg.attach(msg_body)
            msg.attach(attachment)
            s = smtplib.SMTP('localhost')
            s.sendmail(
                CONFIG['ALERT']['from'],
                recipients.replace(' ', '').split(','),
                msg.as_string()
                )
            s.quit()


@cmd.command()
@click.argument('client_name')
def details(client_name):
    """Renvoie le détail de consommation des 30 derniers jours du client"""

    def str_data(conso_bytes):
        conso_megabytes = round(
            (float(conso_bytes) / 1000000),
            2
            )
        result = str(conso_megabytes).replace('.', ',')
        if result == '0,0':
            return '0'
        else:
            return result

    client = Customer(client_name)
    cdrs = client.get_cdr_summary(daily_details=True)
    consumption = cdrs['consumption']
    details = consumption['details']
    dataset = tablib.Dataset()
    dataset.headers = [
        'date',
        'HO',
        'data_limit',
        'total_download_national_HO',
        'total_upload_national_HO',
        'total_download_HO',
        'total_upload_HO',
        'total_HO',
        'total_download_national_HNO',
        'total_upload_national_HNO',
        'total_download_HNO',
        'total_upload_HNO',
        'total_HNO'
       ]
    for daily_detail in details:
        click.echo(daily_detail)
        click.echo('\n')
        date = daily_detail['date']
        daily_total_download = daily_detail['octets_in']
        daily_total_upload = daily_detail['octets_out']
        daily_total = daily_detail['total']
        daily_cdr_by_source = daily_detail['details']
        daily_cdr_by_source = {
            i['source']: {
                'octets_in': i['octets_in'],
                'octets_out': i['octets_out'],
                'total': i['total']
                }
            for i in daily_cdr_by_source
            }
        daily_total_download_national = daily_cdr_by_source.get(
            'national', {}).get(
                'octets_in', 0
                )
        daily_total_upload_national = daily_cdr_by_source.get(
            'national', {}).get(
                'octets_out', 0
                )
        daily_cdrs_policy_wise = daily_detail['policy_wise']
        for info in daily_cdrs_policy_wise:
            policy = info['interval']
            data_limit = info['data_limit']
            info_by_source = {
                i['source']: {
                    'octets_in': i['octets_in'],
                    'octets_out': i['octets_out'],
                    'total': i['total']
                    }
                for i in info['details']
                }
            daily_total_download_for_fup_HO = info['octets_in']
            daily_total_upload_for_fup_HO = info['octets_out']
            daily_total_download_national_HO = info_by_source.get(
                'national', {}).get(
                    'octets_in', 0
                    )
            daily_total_upload_national_HO = info_by_source.get(
                'national', {}).get(
                    'octets_out', 0
                    )
            daily_total_download_HO = (
                daily_total_download_for_fup_HO +
                daily_total_download_national_HO
                )
            daily_total_upload_HO = (
                daily_total_upload_for_fup_HO +
                daily_total_upload_national_HO
                )
            daily_total_HO = (
                daily_total_download_HO +
                daily_total_upload_HO
                )
            assert(daily_total_download >= daily_total_download_HO)
            daily_total_download_HNO = (
                daily_total_download - daily_total_download_HO
                )
            assert(daily_total_upload >= daily_total_upload_HO)
            daily_total_upload_HNO = (
                daily_total_upload - daily_total_upload_HO
                )
            assert(
                daily_total_download_national >=
                daily_total_download_national_HO
                )
            daily_total_download_national_HNO = (
                daily_total_download_national -
                daily_total_download_national_HO
                )
            assert(
                daily_total_upload_national >=
                daily_total_upload_national_HO
                )
            daily_total_upload_national_HNO = (
                daily_total_upload_national -
                daily_total_upload_national_HO
                )
            daily_total_HNO = daily_total - daily_total_HO
            dataset.rpush(
                [
                    '{}/{}/{}'.format(date[:2], date[2:4], date[4:]),
                    policy,
                    data_limit * 1000,
                    str_data(daily_total_download_national_HO),
                    str_data(daily_total_upload_national_HO),
                    str_data(daily_total_download_HO),
                    str_data(daily_total_upload_HO),
                    str_data(daily_total_HO),
                    str_data(daily_total_download_national_HNO),
                    str_data(daily_total_upload_national_HNO),
                    str_data(daily_total_download_HNO),
                    str_data(daily_total_upload_HNO),
                    str_data(daily_total_HNO)
                    ]
                )
    with open('/tmp/details_{}.csv'.format(client_name), 'w') as f:
        content = dataset.export('csv', delimiter=';')
        f.write(content)
# EOF
