#!/usr/bin/env python
# encoding: utf-8

"""
Ce module contient les briques de bases du service
de gestion des politiques de consommation
des utilisateurs Internet chez Blueline.

Une synchronisation des données est effectuée tous les matins
avec Aiguillier pour disposer d'une base locale afin de faire
des calculs sans craindre les ralentissements réseau. Toutes
les vérifications se font donc en local.
"""
import logging
import smtplib
import json
from email.mime.base import MIMEBase

import requests

from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
from email.utils import formatdate

from .consts import FUPDB_CREDENTIALS, APIGW, LOGS, CONFIG

import records
import psycopg2


__all__ = [
    'database_url',
    'record_customer',
    'apply_product_policies',
    'update_customer_cdr'
    ]


def connexion(**kwargs):
    con = psycopg2.connect(
        user=kwargs.get('user'),
        password=kwargs.get('password'),
        host=kwargs.get('host'),
        port=kwargs.get('port'),
        database=kwargs.get('name')
    )
    return con


def database_url(**kwargs):
    url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
        eng=kwargs.get('engine'),
        user=kwargs.get('user'),
        password=kwargs.get('password'),
        host=kwargs.get('host'),
        port=kwargs.get('port'),
        name=kwargs.get('name')
        )
    return url


def si_database_url():
    url = '{eng}://{user}:{password}@{host}:{port}/{name}'.format(
        eng=CONFIG['DB']['engine'],
        user=CONFIG['DB']['user'],
        password=CONFIG['DB']['password'],
        host=CONFIG['DB']['host'],
        port=CONFIG['DB']['port'],
        name=CONFIG['DB']['name'],
    )
    return url


def get_fuped_aiguiller():
    """
    Fetches all fuped customers in Aiguiller.
    The called API returns both BI and SI customers and results need to be
    filtered accordingly.
    """
    try:
        response = requests.get(
            CONFIG['AIGUILLIER']['url'] + '/fup',
            auth=(
                CONFIG['AIGUILLIER']['username'],
                CONFIG['AIGUILLIER']['password'],
            )
        )
        return response.json()['fup']
    except Exception as e:
        operation = "get fuped customers in Aiguiller"
        logging.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        return []


def get_customers_si(identifier=None, value=None):
    """Fetches fup-able customers in fup-service database."""
    if identifier and value:
        sql = (
            "SELECT * FROM customer where {}='{}'"
            .format(identifier, value)
        )
    else:
        sql = "SELECT * FROM customer"
    try:
        with records.Database(si_database_url()) as db:
            rows = db.query(sql)
            customer_l = rows.as_dict()
            return customer_l
    except Exception as e:
        operation = "fetch customers list in SI database"
        logging.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        return []


def get_apigw_token():
    """
    Fetches authentication token for requests to send to API GATEWAY services.
    """
    try:
        token_response = requests.post(
            CONFIG['APIGATEWAY']['token_url'],
            auth=(
                CONFIG['APIGATEWAY']['username'],
                CONFIG['APIGATEWAY']['password']
            ),
        )
        logging.info(
            "Get api gateway authentication token: {}"
            .format(token_response.status_code)
        )
        return token_response.json()['token']
    except Exception as e:
        operation = "get api gateway token"
        logging.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        raise e


def get_customers_crm(product_name):
    """
    Fetches all customers information from CRM(4D) with their products.
    Only business_access, business_plus and airfiber (aka airfiber_prime)
    will be returned by this API.
    :param product_name: ["business_access", "business_plus", "airfiber"]
    :return: list of customers owning the provided product and
    their information
    """
    token = get_apigw_token()
    try:
        response = requests.post(
            CONFIG['APIGATEWAY']['crm_customers_url'],
            headers={'Content-Type': 'application/json', },
            data=json.dumps({'type': product_name, 'version': '2'}),
            auth=(token, ''),
        )
        logging.info(
            "Get 4D customer list for product '{}' : {}"
            .format(product_name, response.status_code)
        )
        return response.json()['root']['data']['list']['produit']
    except Exception as e:
        operation = "get 4D customer list"
        logging.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )


def get_last_history(customer_name):
    """
    Get a given customer's last fup history in fup-services database.
    :param customer_name: name of the customer, already lowered
    """
    try:
        with records.Database(si_database_url()) as db:
            sql = (
                "SELECT * FROM fup "
                "WHERE datetime = "
                "(SELECT MAX(datetime) FROM fup "
                "WHERE customer='{name}') "
            ).format(name=customer_name)
            rows = db.query(sql)
            fup_history = rows.as_dict()
        return fup_history[0]
    except Exception as e:
        operation = (
            "get customer {}'s last fup history in SI database"
            .format(customer_name)
        )
        logging.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )
        return []

# =============================== customers ==================================


def sync_customer(name, product_name, refnum):
    """
    Synchronizes (create or update) the customer with its instance in database.
    """
    exists = get_customers_si(
        identifier="name",
        value=name
    )[0]
    if exists:
        if exists['refnum'] != refnum or exists['product'] != product_name:
            update_customer(name, product_name, refnum)
    else:
        create_customer(name, product_name, refnum)


def update_customer(customer_name, product_name, primary_ref):
    """
    Updates a customer in fup-services database
    """
    sql = (
        "UPDATE customer SET refnum={ref}, product={product}"
        "WHERE name={name};"
        .format(ref=primary_ref, name=customer_name, product=product_name)
    )
    try:
        with records.Database(si_database_url()) as db:
            db.query(sql)
        logging.info(
            "Customer {}[{}] with product {} updated in SI database"
            .format(customer_name, primary_ref, product_name)
        )
    except Exception as e:
        operation = (
            "update customer {} in SI database"
            .format(customer_name)
        )
        logging.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )


def create_customer(customer_name, product_name, primary_ref):
    """
    Creates a new customer entry in fup-services database.
    """
    sql = (
        "INSERT INTO customer(refnum, name, product)"
        "VALUES('{ref}', '{name}', '{product}');"
        .format(ref=primary_ref, name=customer_name, product=product_name)
    )
    try:
        with records.Database(si_database_url()) as db:
            db.query(sql)
        logging.info(
            "Customer {}[{}] with product {} created in SI database"
            .format(customer_name, primary_ref, product_name)
        )
    except Exception as e:
        operation = (
            "create customer {} in SI database"
            .format(customer_name)
        )
        logging.error("Unable to {}".format(operation))
        send_error_report(
            operation=operation,
            error=str(e)
        )


def purge_customer(name):
    """
    Remove all information stored in database about the given customer
    """
    url = database_url(**FUPDB_CREDENTIALS)
    with records.Database(url) as db:
        sql = "DELETE FROM customer WHERE name=:name"
        db.query(sql, name=name)
        sql = "DELETE FROM cdr WHERE customer=:name"
        db.query(sql, name=name)
        sql = "DELETE FROM fup WHERE customer=:name"
        db.query(sql, name=name)


def record_customer(infos):
    """
    Add the given customer and its information to database
    """
    url = database_url(**FUPDB_CREDENTIALS)
    with records.Database(url) as db:
        sql = (
            "INSERT INTO customer (name, refnum, product) "
            "VALUES (:name, :refnum, :product)"
            )
        db.query(
            sql,
            name=infos['name'],
            refnum=infos['refnum'],
            product=infos['product']
            )


# ================================ Products ==================================

def get_product_policy(product):
    """
    Fetches the policy information related to the product.
    :param product: the product name
    :return : {
        'max_data': maximum consumption allowed,
        'start_time': consumption starting time (HHmmss),
        'end_time': consumption ending time (HHmmss),
        'start_date': consumption starting day (isodow),
        'end_date': consumption ending day (isodow),
    }
    """
    policy_raw = CONFIG['POLICIES'][product].split(',')
    policy = {
        'max_data': policy_raw[0],
        'start_time':  policy_raw[1],
        'end_time': policy_raw[2],
        'start_date': policy_raw[3],
        'ending_date': policy_raw[4]
    }
    return policy


# =============================== CDRs =====================================

def update_consommation(product, policy, customer, date, id, total):
    sql = (
        "INSERT INTO fup(product, policy, customer, datetime, status, id, conso)"
        "VALUES (?, ?, ?, ?, ?, ?, ?);"
    )


def get_consommation_sum():
    sql = (
        "SELECT SUM(octets_in+octets_out)"
        "FROM cdr"
        "WHERE customer ilike '%{customer}%'"
        "AND datetime >= '{date}'::date"  # YYYY-MM-DD
        "AND source='internet'"
        "AND (extract('ISODOW' FROM datetime) between {start_date} AND {end_date})"  # dynamic based on policy
        "AND extract(HOUR FROM datetime) < 19"
        "AND extract(HOUR FROM datetime) > 8"
        "GROUP BY customer;"
    )


def apply_product_policies(for_real, customer):
    """Sous routine appliquant les politiques de bon usage
    à un client en particulier.
    Utile pour ensuite faire de l'exécution parallèle"""
    response = customer.apply_product_policies(for_real=for_real)
    data = [
        policy for policy in response
        # if 'fup_in' in policy or 'fup_out' in policy
        ]
    return customer, data


def update_customer_cdr(last_date, customer):
    """Mise à jour des CDRs d'un client dans la base REDIS"""
    customer.get_cdr_from_aiguillier(
        last_date=last_date
        )


def send_error_report(operation, error):
    """
    Sends an email to development and monitoring team to notify an error
    in the process.
    """
    message = (
        "Service: {} [{}]\n\n"
        "Affected operation: {}\n\n"
        "Error: {}"
        .format("fup-services", APIGW, operation, error)
    )
    sender = "api-gateway@si.blueline.mg"
    # destinations = ["dev@si.blueline.mg", "sysadmin@si.blueline.mg"]
    destinations = ["yona.rasolonjatovo@staff.blueline.mg",]
    content = MIMEText(message, 'plain', 'utf-8')

    msg = MIMEMultipart()
    msg['Subject'] = "fup-services process error"
    msg['From'] = sender
    msg['To'] = ', '.join(destinations)
    msg['Date'] = formatdate(localtime=True)
    msg.attach(content)

    smtp = smtplib.SMTP()
    smtp.connect("smtp.blueline.mg", port=10026)
    # smtp = smtplib.SMTP('localhost')
    smtp.sendmail(sender, destinations, msg.as_string())
    smtp.quit()

    LOGS.logger.info(
        "Email: '{}' sent to {}".format(msg['Subject'], destinations))
    smtp.quit()
    # message = (
    #     "Service: {} [{}]\n\n"
    #     "Affected operation: {}\n\n"
    #     "Error: {}"
    #     .format("fup-services", APIGW, operation, error)
    # )
    # destinations = ["dev@si.blueline.mg", "sysadmin@si.blueline.mg"]
    #
    # send_email(
    #     destinations=destinations,
    #     subject="fup-services process error",
    #     message=message
    # )


def send_email(destinations, subject, message, attachment=None):
    """
    Generic function for sending emails.
    :param destinations: single email or list of emails
    :param subject: email subject
    :param message: email content
    :attachment: files that will be attached to the email if any
    """
    sender = "api-gateway@si.blueline.mg"
    destinations = "yona.rasolonjatovo@staff.blueline.mg"
    if not isinstance(destinations, list):
        destinations = list(destinations)
    content = MIMEText(message, 'plain', 'utf-8')

    # if attachment:
    #     msg = MIMEMultipart('related')
    #     attachment = MIMEBase('text', 'csv')
    #     msg.attach(attachment)
    #     fp = open('/tmp/details_reconciliation.csv', 'rb')
    #     attachment.set_payload(fp.read())
    #     fp.close()
    #     attachment.add_header(
    #         'Content-Disposition',
    #         'attachment',
    #         filename='details_reconciliation.csv'
    #     )
    # else:
    #     msg = MIMEMultipart()
    msg = MIMEMultipart()
    msg['Subject'] = subject
    msg['From'] = sender
    msg['To'] = ', '.join(destinations)
    msg['Date'] = formatdate(localtime=True)
    msg.attach(content)

    smtp = smtplib.SMTP('localhost')
    smtp.sendmail(sender, destinations, msg.as_string())
    smtp.quit()

    LOGS.logger.info(
        "Email: '{}' sent to {}".format(msg['Subject'], destinations))
    smtp.quit()

# EOF
