# -*- coding: utf-8 -*-

from flask import request

from blueflask.lib.infos import created, success
from blueflask.lib.errors import conflict, internal_error, not_found

from ...consts import SERVICE_CODE
from ...core import Client
from . import api


@api.route('/subscribers/imsi/<string:isdn>', methods=['GET'])
def list_imsi(isdn):
    """
    List IMSI of an ISDN.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: HLR response
        schema:
          id: HLR response
          properties:
            code:
              type: integer
              description: telnet code
            data:
              description: list of infos
              type: array
              items:
                type: string
            date:
              description: date of response
              type: string
            time:
              description: time of response
              type: string
            message:
              description: summary of response
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    client = Client()
    client.list_imsi(isdn)
    code = client.response['code']
    message = client.response['message']
    return_code = '{service_code}-{code}'.format(
        service_code=SERVICE_CODE,
        code=code
        )
    if code == 3001:
        return not_found(
            message=message,
            code=return_code,
            service_code=SERVICE_CODE
            )
    elif code:
        return internal_error(
            message=message,
            code=return_code,
            service_code=SERVICE_CODE
            )
    else:
        return success(service_code=SERVICE_CODE, message=client.response)


@api.route('/subscribers/isdn/<string:imsi>', methods=['GET'])
def list_isdn(imsi):
    """
    List ISDN of an IMSI.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: HLR response
        schema:
          id: HLR response
          properties:
            code:
              type: integer
              description: telnet code
            data:
              description: list of infos
              type: array
              items:
                type: string
            date:
              description: date of response
              type: string
            time:
              description: time of response
              type: string
            message:
              description: summary of response
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    client = Client()
    client.list_isdn(imsi)
    code = client.response['code']
    message = client.response['message']
    return_code = '{service_code}-{code}'.format(
        service_code=SERVICE_CODE,
        code=code
        )
    if code == 3001:
        return not_found(
            message=message,
            code=return_code,
            service_code=SERVICE_CODE
            )
    elif code:
        return internal_error(
            message=message,
            code=code,
            service_code=SERVICE_CODE
            )
    else:
        return success(service_code=SERVICE_CODE, message=client.response)


@api.route('/subscribers/isdn', methods=['PUT'])
def update_isdn():
    """
    Update ISDN.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: HLR response
        schema:
          id: HLR response
          properties:
            code:
              type: integer
              description: telnet code
            data:
              description: list of infos
              type: array
              items:
                type: string
            date:
              description: date of response
              type: string
            time:
              description: time of response
              type: string
            message:
              description: summary of response
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    client = Client()
    data = request.json()
    imsi = data.get('imsi')
    isdn = data.get('isdn')
    if imsi:
        data = {'imsi': imsi}
    elif isdn:
        data = {'isdn': isdn}
    client.update_isdn(**data)
    code = client.response['code']
    message = client.response['message']
    return_code = '{service_code}-{code}'.format(
        service_code=SERVICE_CODE,
        code=code
        )
    if code:
        return internal_error(
            message=message,
            code=return_code,
            service_code=SERVICE_CODE
            )
    return success(service_code=SERVICE_CODE, message=client.response)


@api.route('/subscribers', methods=['POST'])
def create_subscriber():
    """
    Create subscriber in HLR.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      201:
        description: ressource created
        schema:
          id: no content
          properties:
            code:
              type: integer
            message:
              type: string
            info:
              type: string
            status:
              type: string
      409:
        description: conflict
        schema:
          id: conflict
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    client = Client()
    data = request.get_json()
    imsi = data['imsi']
    isdn = data['isdn']
    client.add_subscriber(imsi, isdn)
    code = client.response['code']
    message = client.response['message']
    return_code = '{service_code}-{code}'.format(
        service_code=SERVICE_CODE,
        code=code
        )
    if code:
        if code == 3002:
            return conflict(
                message=message,
                code=return_code,
                service_code=SERVICE_CODE
                )
        else:
            return internal_error(
                message=message,
                code=return_code,
                service_code=SERVICE_CODE
                )
    else:
        client.update_gprs_profile(imsi, True)
        code = client.response['code']
        message = client.response['message']
        return_code = '{service_code}-{code}'.format(
            service_code=SERVICE_CODE,
            code=code
            )
        if code:
            return internal_error(
                message=message,
                code=return_code,
                service_code=SERVICE_CODE
                )
        else:
            return created(service_code=SERVICE_CODE)


@api.route('/subscribers', methods=['DELETE'])
def delete_subscriber():
    """
    Delete subscriber in HLR.
    ---
    tags:
      - subscribers
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: HLR response
        schema:
          id: HLR response
          properties:
            code:
              type: integer
              description: telnet code
            data:
              description: list of infos
              type: array
              items:
                type: string
            date:
              description: date of response
              type: string
            time:
              description: time of response
              type: string
            message:
              description: summary of response
              type: string
      500:
        description: internal error
        schema:
          id: internal error
          properties:
            code:
              type: integer
            message:
              type: string
            error:
              type: string
            status:
              type: string
    """
    client = Client()
    data = request.get_json()
    client.remove_subscriber(**data)
    code = client.response['code']
    message = client.response['message']
    return_code = '{service_code}-{code}'.format(
        service_code=SERVICE_CODE,
        code=code
        )
    if code:
        return internal_error(
            message=message,
            code=return_code,
            service_code=SERVICE_CODE
            )
    return success(service_code=SERVICE_CODE, message=client.response)

# EOF
