# -*- coding: utf-8 -*-

from datetime import datetime

from .consts import CONFIG, LOGS

import telnetlib

__all__ = [
    'Client'
    ]


class Client(object):
    """
    Wrapper autour de telnetlib pour effectuer des requêtes
    de façon plus efficace.
    Permettra par la suite de faire de l'asynchrone.
    """

    def __init__(self):
        self.host = CONFIG['HLR']['host']
        self.port = int(CONFIG['HLR']['port'])
        self.username = CONFIG['HLR']['username']
        self.timeout = int(CONFIG['HLR']['timeout'])

    @property
    def response(self):
        """Parsing de la réponse du HLR"""

        response = [line for line in self._response.split('\n') if line]
        date, time = response[0].split()[-2:]
        result = response[3].replace(' = ', '=')
        retcode = int(result.split()[0].split('=')[1])
        details = ' '.join(result.split()[1:])
        data = [i.replace(' ', '') for i in response[4:-2]]
        return {
            'date': date,
            'time': time,
            'code': retcode,
            'message': details,
            'data': data
            }

    @response.setter
    def response(self, x):
        self._response = x

    def connect(self):
        return telnetlib.Telnet(self.host, self.port, self.timeout)

    def send_cmd(self, telnet, cmd):
        if cmd.startswith('LGI'):
            LOGS.logger.info('> LGI')
            LOGS.logger.debug(
                '{}, {}'.format(
                    cmd.split(', ')[0],
                    ', '.join(cmd.split(', ')[2:])
                    )
                )
        else:
            LOGS.logger.info('> {}'.format(cmd.split(':')[0]))
            LOGS.logger.debug(cmd)
        cmd = '{};'.format(cmd)
        telnet.write(cmd.encode('utf-8'))
        # La réponse du serveur telnet du HLR ressemble à ceci:
        #
        # +++    USCDB        YYYY-mm-dd H:M:S
        # PGW    #<some_id>
        # %% <the_text_of_the_sent_cmd>%%
        # RETCODE = <retcode> <some_details>
        #
        # <result_displayed_here_if_needed>
        #
        # ---    END
        index, regex_match, text = telnet.expect(
            [],
            int(CONFIG['HLR']['timeout_per_request'])
            )
        if not text:
            LOGS.logger.error('No response received')
            self._response = (
                '+++    USCDB        {date_and_time}\n'
                'PGW    #061780\n'
                '%%{cmd}%%\n'
                'RETCODE = {retcode} {details}\n\n'
                '{data}\n\n'
                'There is together 1 report\n\n'
                '---    END\n'
                ).format(
                    date_and_time=datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
                    cmd=cmd,
                    retcode=504,
                    details='gateway timeout',
                    data="Le serveur telnet du HLR a renvoyé une réponse vide"
                    )
        else:
            self._response = text.decode('utf-8').replace('\r', '')
        LOGS.logger.debug(self._response)
        return self._response

    def send(self, cmd):
        log_in = (
            'LGI: OPNAME="{username}", '
            'PWD="{password}", '
            'HLRSN=1, '
            'IP="41.204.98.34"'
            ).format(
                username=self.username,
                password=CONFIG['HLR']['password']
                )
        log_out = 'LGO:'
        telnet = self.connect()
        self.send_cmd(telnet, log_in)
        response = self.send_cmd(telnet, cmd)
        self.send_cmd(telnet, log_out)
        self.response = response
        telnet.close()
        return response

    def list_imsi(self, isdn):
        cmd = 'LST IMSI: ISDN="{}"'.format(isdn)
        return self.send(cmd)

    def list_isdn(self, imsi):
        cmd = 'LST ISDN: IMSI="{}"'.format(imsi)
        return self.send(cmd)

    def add_subscriber(self, imsi, isdn):
        cmd = 'ADD TPLSUB: HLRSN=1, IMSI="{}", ISDN="{}", TPLID=110'.format(
            imsi,
            isdn
            )
        self.send(cmd)
        cmd = 'ACT CWAIT: ISDN="{}"'.format(isdn)
        return self.send(cmd)

    def update_gprs_profile(self, imsi, flag=True):
        if flag:
            flag = 'TRUE'
        else:
            flag = 'FALSE'
        cmd = 'MOD TPLGPRS: IMSI="{}", PROV={}, TPLID=111'.format(imsi, flag)
        return self.send(cmd)

    def remove_subscriber(self, **kwargs):
        cmd = 'RMV SUB: {}="{}", RMVKI=TRUE'
        if 'imsi' in kwargs:
            cmd = cmd.format('IMSI', kwargs['imsi'])
        elif 'isdn' in kwargs:
            cmd = cmd.format('ISDN', kwargs['isdn'])
        else:
            raise ValueError('You must provide either an imsi or an isdn')

        return self.send(cmd)

    def update_isdn(self, new_isdn, **kwargs):
        cmd = 'MOD ISDN: {}="{}", NEWISDN="{}"'
        if 'imsi' in kwargs:
            cmd = cmd.format('IMSI', kwargs['imsi'], new_isdn)
        elif 'isdn' in kwargs:
            cmd = cmd.format('ISDN', kwargs['isdn'], new_isdn)
        else:
            raise ValueError('You must provide either an imsi or an isdn')

        return self.send(cmd)

    def block_simbox(self, **kwargs):
        cmd = 'MOD ODB: {}="{}", ODBIC=BAIC, ODBOC=BAOC, ODBPOS=BAPOS'
        if 'imsi' in kwargs:
            cmd = cmd.format('IMSI', kwargs['imsi'])
        elif 'isdn' in kwargs:
            cmd = cmd.format('ISDN', kwargs['isdn'])
        else:
            raise ValueError('You must provide either an imsi or an isdn')

        return self.send(cmd)

    def free_simbox(self, **kwargs):
        cmd = 'MOD ODB: {}="{}", ODBIC=NOBIC, ODBOC=NOBOC, ODBPOS=NOBPOS'
        if 'imsi' in kwargs:
            cmd = cmd.format('IMSI', kwargs['imsi'])
        elif 'isdn' in kwargs:
            cmd = cmd.format('ISDN', kwargs['isdn'])
        else:
            raise ValueError('You must provide either an imsi or an isdn')

        return self.send(cmd)

# EOF
