#!/usr/bin/env python
# encoding: utf-8

from flask import request

from blueflask.lib.decorators import auth as ldap
from blueflask.lib.errors import bad_request, not_found
from blueflask.lib.errors import internal_error, unauthorized
from blueflask.lib.infos import success

from ... import REDIS_DB, SERVICE_CODE
from ... import core
from . import api


@api.route('/auth/verify-code', methods=['POST'])
@ldap.login_required
def verify_code():
    """
    Vérification du code secret d'un abonné.
    ---
    tags:
      - Secret Code
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: string
              description: Status de la requête HTTP
            code:
              type: string
              description: Code d'erreur interne à l'API
            error:
              type: string
              description: message court décrivant l'erreur
            message:
              type: string
              description: message décrivant en détail l'erreur
    parameters:
      - name: phonenumber
        in: body
        type: string
        required: true
        description: Numéro de téléphone
      - name: code
        in: body
        type: string
        required: true
        description: Nouveau code
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: La vérification a marché
      400:
        description: Les paramètres en entrée sont incorrects
        $ref: "#/definitions/Error"
      401:
        description: La vérification a échoué
        $ref: "#/definitions/Error"
    """
    data = request.get_json() or {}
    if not data or 'phonenumber' not in data or 'code' not in data:
        return bad_request(service_code=SERVICE_CODE)
    elif core.secret_code(data['phonenumber'], data['code'], flag=True):
        return success(service_code=SERVICE_CODE)
    else:
        return unauthorized(
            message='Le code secret est incorrect',
            code='{root}-{code}'.format(
                root=SERVICE_CODE,
                code='4010'
                ),
            service_code=SERVICE_CODE
            )


@api.route('/auth', methods=['POST'])
@ldap.login_required
def modify_code():
    """
    Modification du code secret.
    ---
    tags:
      - Secret Code
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: string
              description: Status de la requête HTTP
            code:
              type: string
              description: Code d'erreur interne à l'API
            error:
              type: string
              description: message court décrivant l'erreur
            message:
              type: string
              description: message décrivant en détail l'erreur
    parameters:
      - name: phonenumber
        in: body
        type: string
        required: true
        description: Numéro de téléphone
      - name: code
        in: body
        type: string
        description: Nouveau code
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: La modification a marché
      400:
        description: Les paramètres en entrée sont incorrects
        $ref: "#/definitions/Error"
      500:
        description: Erreur interne
        $ref: "#/definitions/Error"
    """
    data = request.get_json() or {}
    if not data or 'phonenumber' not in data:
        return bad_request(service_code=SERVICE_CODE)
    else:
        try:
            core.secret_code(data['phonenumber'], data.get('code'))
        except Exception:
            return internal_error(service_code=SERVICE_CODE)
        else:
            return success(service_code=SERVICE_CODE)


@api.route('/auth/<string:phonenumber>', methods=['GET'])
@ldap.login_required
def has_code(phonenumber):
    """
    Vérification si l'abonné a un code secret.
    ---
    tags:
      - Secret Code
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: string
              description: Status de la requête HTTP
            code:
              type: string
              description: Code d'erreur interne à l'API
            error:
              type: string
              description: message court décrivant l'erreur
            message:
              type: string
              description: message décrivant en détail l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: L'utilisateur a un code
      404:
        description: L'utilisateur n'a pas de code
        $ref: "#/definitions/Error"
    """
    if not REDIS_DB.get(phonenumber):
        return not_found(
            message="L'utilisateur n'a pas de code",
            code='{root}-{code}'.format(
                root=SERVICE_CODE,
                code='4040'
                )
            )
    else:
        return success(service_code=SERVICE_CODE)

# EOF
