# -*- coding: utf-8 -*-

from flask import request

from blueflask.lib.errors import bad_request, internal_error, not_found
from blueflask.lib.infos import success

from ...consts import CONFIG, SERVICE_CODE
from ... import adjust_account, cash_recharge
from ... import sub_mgrt
from ... import query
from . import api

import itertools


@api.route("/subscribers/<string:phonenumber>", methods=["PUT"])
def modify_subscriber_informations(phonenumber):
    """
    Modification des informations d'un utilisateur
    ---
    tags:
      - Subscribers
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: string
              description: Status de la requête HTTP
            code:
              type: string
              description: Code d'erreur interne à l'API
            error:
              type: string
              description: message court décrivant l'erreur
            message:
              type: string
              description: message décrivant en détail l'erreur
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: Changements OK
      400:
        description: Le format du numéro est incorrect
        $ref: "#/definitions/Error"
      404:
        description: Impossible de trouver le numéro
        $ref: "#/definitions/Error"
      500:
        description: Erreur interne à l'API
        $ref: "#/definitions/Error"
    """

    user = request.authorization.username
    interface = sub_mgrt.SubmgrtMgr(user)
    data = request.get_json()
    if "language_code" in data:
        data["SMSLangType"] = data["language_code"]
        data["LangType"] = data["language_code"]
        del data["language_code"]
    code, result = interface.ChangeSubscriberBasicInfor(phonenumber[-9:], **data)
    return_code = "{service_code}-{code}".format(service_code=SERVICE_CODE, code=code)
    if code == "405000000":
        return success(service_code=SERVICE_CODE)
    elif code == "102010144":
        return bad_request(service_code=SERVICE_CODE, code=return_code)
    else:
        return internal_error(service_code=SERVICE_CODE, code=return_code)


@api.route("/subscribers/<string:phonenumber>", methods=["GET"])
def get_subscriber_informations(phonenumber):
    """
    Informations complètes sur un utilisateur.
    ---
    tags:
      - Subscribers
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: string
              description: Status de la requête HTTP
            code:
              type: string
              description: Code d'erreur interne à l'API
            error:
              type: string
              description: message court décrivant l'erreur
            message:
              type: string
              description: message décrivant en détail l'erreur
      - schema:
          id: Subscription
          properties:
            name:
              type: string
              description: nom du produit
            balance:
              type: string
              description: extrait du solde de cet abonnement
      - schema:
          id: Informations
          properties:
            phonenumber:
              type: string
              description: numéro de téléphone de l'abonné
            language_code:
              type: integer
              description: code langue abonné
            subscriptions:
              type: array
              items:
                $ref: "#/definitions/Subscription"
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    responses:
      200:
        description: Informations sur le client
        $ref: "#/definitions/Informations"
      204:
        description: Aucun abonnement
      400:
        description: Le format du numéro est incorrect
        $ref: "#/definitions/Error"
      404:
        description: Impossible de trouver le numéro
        $ref: "#/definitions/Error"
      500:
        description: Erreur interne à l'API
        $ref: "#/definitions/Error"
    """
    user = request.authorization.username
    interface = query.QueryMgr(user)

    def get_language(result):
        """récupération de la langue"""
        lang_type = list(
            filter(
                lambda x: x["Id"] == "SMSLangType",
                result.Subscriber[0].SubscriberInfo.SimpleProperty,
            )
        )
        lang_type = lang_type.pop()
        return int(lang_type["Value"])

    def get_subscriptions(result):
        """récupération des abonnements"""
        response = []
        for account in result.Account:
            balances = sorted(
                account.BalanceRecordList.Balance, key=lambda x: x.RelatedObjectID
            )
            balances_per_ro_id = itertools.groupby(
                balances, key=lambda x: x.RelatedObjectID
            )
            for ro_id, ro_id_balances in balances_per_ro_id:
                if not ro_id:
                    continue
                section = "BALANCE_RELATEDOBJECTID_{}".format(ro_id)
                if section in CONFIG:
                    infos = {"name": CONFIG[section]["name"].replace("_", "'")}
                    ro_id_balances = sorted(ro_id_balances, key=lambda x: x.AccountType)
                    balances_per_act_type = itertools.groupby(
                        ro_id_balances, key=lambda x: x.AccountType
                    )
                    balance_info = []
                    for key, values in balances_per_act_type:
                        # key représente le type de compte
                        # values représentent les soldes de ce compte
                        balance = sum([i.Amount for i in values])
                        unit = CONFIG["ACCOUNT_{}".format(key)]["unit"]
                        if unit == "Mo":
                            balance = round(balance / 1024 / 1024)
                        balance_info.append("{}{}".format(balance, unit))
                    infos["balance"] = ", ".join(balance_info)
                    response.append(infos)
        return response

    code, result = interface.queryCustomer(phonenumber[-9:])
    return_code = "{service_code}-{code}".format(service_code=SERVICE_CODE, code=code)
    if code == "102025033":
        return bad_request(
            message="Numéro de téléphone incorrect",
            code=return_code,
            service_code=SERVICE_CODE,
        )
    elif code in ["102010128", "102010248"]:
        return not_found(
            message="Numéro de téléphone inexistant",
            code=return_code,
            service_code=SERVICE_CODE,
        )
    elif code == "405000000":
        response = {"phonenumber": phonenumber}
        searchword = request.args.get("key", None)
        if searchword == "language_code":
            response["language_code"] = get_language(result)
            return success(service_code=SERVICE_CODE, message=response)
        elif searchword == "subscriptions":
            response["subscriptions"] = get_subscriptions(result)
            return success(service_code=SERVICE_CODE, message=response)
        else:
            response["language_code"] = get_language(result)
            response["subscriptions"] = get_subscriptions(result)
            code, result = interface.queryBalance(phonenumber[-9:], "2000")
            response["subscriptions"].append(
                {"name": "principal", "balance": "{} Ar".format(result["balance"])}
            )
            return success(service_code=SERVICE_CODE, message=response)
    else:
        return internal_error(service_code=SERVICE_CODE, code=return_code)


@api.route("/subscribers/<string:phonenumber>/balances", methods=["GET"])
def get_subscriber_balance(phonenumber):
    """
    Solde d'un compte d'un utilisateur.
    Cette ressource a besoin d'un numéro de téléphone Bip passé en JSON
    afin de récupérer son solde dans l'OCS
    ---
    tags:
      - Subscribers
    definitions:
      - schema:
          id: Error
          properties:
            status:
              type: string
              description: Status de la requête HTTP
            code:
              type: string
              description: Code d'erreur interne à l'API
            error:
              type: string
              description: message court décrivant l'erreur
            message:
              type: string
              description: message décrivant en détail l'erreur
      - schema:
          id: Informations
          properties:
            phonenumber:
              type: string
              description: numéro de téléphone de l'abonné
            balance:
              type: number
              description: solde du compte
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: account_type
        in: body
        type: number
        description: Identifiant OCS du type de compte
      - name: operator
        in: body
        type: string
        description: Identifiant permettant de désigner l'acteur
    responses:
      200:
        description: Solde du numéro de téléphone
        $ref: "#/definitions/Informations"
      404:
        description: Impossible de trouver le numéro de téléphone dans l'OCS
        $ref: "#/definitions/Error"
      400:
        description: Numéro de téléphone incorrect
        $ref: "#/definitions/Error"
      500:
        description: Erreur interne à l'API
        $ref: "#/definitions/Error"
    """
    data = request.get_json() or {}
    account_type = data.get("account_type", "2000")
    if data and data.get("operator"):
        operator_id = data.get("operator")
        del data["operator"]
        user = operator_id
    else:
        user = request.authorization.username
    interface = query.QueryMgr(user)
    code, result = interface.queryBalance(phonenumber[-9:], account_type)
    return_code = "{service_code}-{code}".format(service_code=SERVICE_CODE, code=code)
    if code == "405000000":
        return success(service_code=SERVICE_CODE, message=result)
    elif code == "102025033":
        return bad_request(
            message="Numéro de téléphone incorrect",
            code=return_code,
            service_code=SERVICE_CODE,
        )
    elif code in ["102010128", "102010248"]:
        return not_found(
            message="Numéro de téléphone inexistant",
            code=return_code,
            service_code=SERVICE_CODE,
        )
    else:
        return internal_error(service_code=SERVICE_CODE, code=return_code)


@api.route("/subscribers/<string:phonenumber>/balances", methods=["PUT"])
def update_subscriber_balance(phonenumber):
    """
    Modification du solde d'un compte d'un utilisateur.
    Cette ressource permet d'ajouter ou de soustraire un montant du solde
    d'un numéro Bip dans l'OCS.
    ---
    tags:
      - Balances
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: amount
        in: body
        type: number
        required: true
        description: Montant à ajouter/enlever
      - name: operation_type
        in: body
        type: number
        required: true
        description: Identifiant OCS du type d'action
      - name: account_type
        in: body
        type: number
        required: true
        description: Identifiant OCS du type de compte
    responses:
      200:
        description: L'opération a réussi
      404:
        description: Impossible de trouver le numéro de téléphone dans l'OCS
      400:
        description: Numéro de téléphone incorrect
      500:
        description: Erreur interne à l'API
    """
    data = request.get_json() or {}
    if data and data.get("operator"):
        operator_id = data.get("operator")
        del data["operator"]
        user = operator_id
    else:
        user = request.authorization.username
    condition = (
        ("amount" in data) and ("account_type" in data) and ("operation_type" in data)
    )
    if not condition:
        return bad_request(service_code=SERVICE_CODE)
    else:
        amount = int(data["amount"])
        del data["amount"]
        account_type = int(data["account_type"])
        del data["account_type"]
        operation_type = int(data["operation_type"])
        del data["operation_type"]
        interface = adjust_account.AdjustAccountMgr(user)
        code, result = interface.AdjustAccount(
            phonenumber[-9:], operation_type, accounts={account_type: amount}, **data
        )
        return_code = "{service_code}-{code}".format(
            service_code=SERVICE_CODE, code=code
        )
        if code == "405000000":
            return success(service_code=SERVICE_CODE)
        elif code == "102025033":
            return bad_request(
                message="Numéro de téléphone incorrect",
                code=return_code,
                service_code=SERVICE_CODE,
            )
        elif code == "102010128":
            return not_found(
                message="Numéro de téléphone inexistant",
                code=return_code,
                service_code=SERVICE_CODE,
            )
        else:
            return internal_error(service_code=SERVICE_CODE, code=return_code)


@api.route("/subscribers/<string:phonenumber>/balances", methods=["POST"])
def charging_subscriber_balance(phonenumber):
    """
    Modification du solde d'un compte d'un utilisateur.
    Cette ressource permet de recharger un montant du solde principal
    d'un numéro Bip dans l'OCS en utilisant l'interface cash_recharge.
    ---
    tags:
      - Balances
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    parameters:
      - name: amount
        in: body
        type: number
        required: true
        description: Montant à ajouter/enlever
      - name: operation_type
        in: body
        type: number
        required: true
        description: Identifiant OCS du type d'action
      - name: account_type
        in: body
        type: number
        required: true
        description: Identifiant OCS du type de compte
    responses:
      200:
        description: L'opération a réussi
      404:
        description: Impossible de trouver le numéro de téléphone dans l'OCS
      400:
        description: Numéro de téléphone incorrect
      500:
        description: Erreur interne à l'API
    """
    data = request.get_json() or {}
    if data and data.get("operator"):
        operator_id = data.get("operator")
        del data["operator"]
        user = operator_id
    else:
        user = request.authorization.username
    condition = "amount" in data
    if not condition:
        return bad_request(service_code=SERVICE_CODE)
    else:
        amount = int(data["amount"])
        receiver = phonenumber[-9:]
        sender = data["sender"]
        del data["sender"]
        del data["amount"]
        interface = cash_recharge.CashRechargeMgr(user)
        code, result = interface.Payment(sender, receiver, amount, **data)
        return_code = "{service_code}-{code}".format(
            service_code=SERVICE_CODE, code=code
        )
        if code == "405000000":
            return success(service_code=SERVICE_CODE)
        elif code == "102025033":
            return bad_request(
                message="Numéro de téléphone incorrect",
                code=return_code,
                service_code=SERVICE_CODE,
            )
        elif code == "102010128":
            return not_found(
                message="Numéro de téléphone inexistant",
                code=return_code,
                service_code=SERVICE_CODE,
            )
        else:
            return internal_error(service_code=SERVICE_CODE, code=return_code)


# EOF
