#!/usr/bin/env python
# encoding: utf-8

# from blueflask.lib.decorators import auth as ldap
from flask import jsonify, request
from blueflask.lib.errors import bad_request
from ...vacations import VacationRequest
from ...consts import SERVICE_CODE
from . import api


@api.route('/vacations/requests', methods=['POST'])
def vacation_request():
    """
    Création congé.
    Ce endpoint permet de créer une damande de congé d'un employé.
    ---
    tags:
      - vacation
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    definitions:
      - schema:
          id: Error message
          properties:
            status:
              type: string
              description: code d'erreur HTTP
            code:
              type: number
              description: code d'erreur d'application
            error:
              type: string
              description: message court décrivant le type d'erreur
            message:
              type: string
              description: détails sur l'erreur rencontrée
      - schema:
          id: Info message
          properties:
            status:
              type: string
              description: code d'info HTTP
            code:
              type: number
              description: code d'info d'application
            info:
              type: string
              description: message court décrivant le type d'info
            message:
              type: string
              description: détails
    responses:
      202:
        description: La création de l'emplacement est passé
        $ref: "#/definitions/Info message"
      400:
        description: La requête est incorecte
        $ref: "#/definitions/Error message"
      401:
        description: Authentification incorects
        $ref: "#/definitions/Error message"
      500:
        description: L'API a recontré une erreur
        $ref: "#/definitions/Error message"
    """
    data = request.get_json()
    date_from = data['date_from']
    date_to = data['date_to']
    employee_id = data['employee_id']
    name = data['name']
    note = data['note']
    response = VacationRequest.create(
        date_from,
        date_to,
        employee_id,
        name,
        note
        )
    if not response:
        return bad_request(
            'Provide a valid JSON document',
            service_code=SERVICE_CODE
            )
    else:
        result = {
            'request_id': response['content']['request_id'],
            'employee_id': data['employee_id']
            }
        return jsonify(result)


@api.route(
    '/vacations/<string:employee_id>',
    methods=['GET']
)
def balance(employee_id):
    """
    Get balance.
    Ce endpoint permet d'avoir le solde de congé d'une personne.
    ---
    tags:
      - vacation
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    definitions:
      - schema:
          id: Error message
          properties:
            status:
              type: string
              description: code d'erreur HTTP
            code:
              type: number
              description: code d'erreur d'application
            error:
              type: string
              description: message court décrivant le type d'erreur
            message:
              type: string
              description: détails sur l'erreur rencontrée
      - schema:
          id: Info message
          properties:
            status:
              type: string
              description: code d'info HTTP
            code:
              type: number
              description: code d'info d'application
            info:
              type: string
              description: message court décrivant le type d'info
            message:
              type: string
              description: détails
    responses:
      202:
        description: La création de l'emplacement est passé
        $ref: "#/definitions/Info message"
      400:
        description: La requête est incorecte
        $ref: "#/definitions/Error message"
      401:
        description: Authentification incorects
        $ref: "#/definitions/Error message"
      500:
        description: L'API a recontré une erreur
        $ref: "#/definitions/Error message"
    """
    response = VacationRequest.get_balance(employee_id)
    return jsonify(response['content'])


@api.route('/vacations/by_matricule/<string:matricule>', methods=['GET'])
def vacations_by_matricule(matricule):
    """
    Avoir la balance par le matricule.
    Ce endpoint permet d'avoir le solde de congé \
    d'une personne en utisant le metricule.
    ---
    tags:
      - vacation
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    definitions:
      - schema:
          id: Error message
          properties:
            status:
              type: string
              description: code d'erreur HTTP
            code:
              type: number
              description: code d'erreur d'application
            error:
              type: string
              description: message court décrivant le type d'erreur
            message:
              type: string
              description: détails sur l'erreur rencontrée
      - schema:
          id: Info message
          properties:
            status:
              type: string
              description: code d'info HTTP
            code:
              type: number
              description: code d'info d'application
            info:
              type: string
              description: message court décrivant le type d'info
            message:
              type: string
              description: détails
    responses:
      202:
        description: La création de l'emplacement est passé
        $ref: "#/definitions/Info message"
      400:
        description: La requête est incorecte
        $ref: "#/definitions/Error message"
      401:
        description: Authentification incorects
        $ref: "#/definitions/Error message"
      500:
        description: L'API a recontré une erreur
        $ref: "#/definitions/Error message"
    """
    response = VacationRequest.balance_by_matricule(matricule)
    return jsonify(response['content'])


@api.route('/vacations/team_balance/<string:employee_id>', methods=['POST'])
def team_balance(employee_id):
    """
    Avoir le solde de congé de son équipe.
    Ce endpoint permet d'avoir le solde de congé \
    d'une personne et son équipe.
    ---
    tags:
      - vacation
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    definitions:
      - schema:
          id: Error message
          properties:
            status:
              type: string
              description: code d'erreur HTTP
            code:
              type: number
              description: code d'erreur d'application
            error:
              type: string
              description: message court décrivant le type d'erreur
            message:
              type: string
              description: détails sur l'erreur rencontrée
      - schema:
          id: Info message
          properties:
            status:
              type: string
              description: code d'info HTTP
            code:
              type: number
              description: code d'info d'application
            info:
              type: string
              description: message court décrivant le type d'info
            message:
              type: string
              description: détails
    responses:
      202:
        description: La création de l'emplacement est passé
        $ref: "#/definitions/Info message"
      400:
        description: La requête est incorecte
        $ref: "#/definitions/Error message"
      401:
        description: Authentification incorects
        $ref: "#/definitions/Error message"
      500:
        description: L'API a recontré une erreur
        $ref: "#/definitions/Error message"
    """
    response = VacationRequest.team_balance(employee_id)
    return jsonify(response['content'])


@api.route(
    '/vacations/requests/<string:requests_id>',
    methods=['GET']
)
def vacation_status(requests_id):
    """
    Savoir l'état d'une demande.
    Ce endpoint permet de savoir si le congé est validé ou pas.
    ---
    tags:
      - vacation
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    definitions:
      - schema:
          id: Error message
          properties:
            status:
              type: string
              description: code d'erreur HTTP
            code:
              type: number
              description: code d'erreur d'application
            error:
              type: string
              description: message court décrivant le type d'erreur
            message:
              type: string
              description: détails sur l'erreur rencontrée
      - schema:
          id: Info message
          properties:
            status:
              type: string
              description: code d'info HTTP
            code:
              type: number
              description: code d'info d'application
            info:
              type: string
              description: message court décrivant le type d'info
            message:
              type: string
              description: détails
    responses:
      202:
        description: La création de l'emplacement est passé
        $ref: "#/definitions/Info message"
      400:
        description: La requête est incorecte
        $ref: "#/definitions/Error message"
      401:
        description: Authentification incorects
        $ref: "#/definitions/Error message"
      500:
        description: L'API a recontré une erreur
        $ref: "#/definitions/Error message"
    """
    vacations = VacationRequest(requests_id)
    response = vacations.status()
    return jsonify(response['content'])


@api.route(
    '/vacations/delete/<string:requests_id>',
    methods=['DELETE']
)
def cancel_vacation(requests_id):
    """
    Annuler une demande.
    Ce endpoint permet d'annuler une demande de congé précis.
    ---
    tags:
      - vacation
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    definitions:
      - schema:
          id: Error message
          properties:
            status:
              type: string
              description: code d'erreur HTTP
            code:
              type: number
              description: code d'erreur d'application
            error:
              type: string
              description: message court décrivant le type d'erreur
            message:
              type: string
              description: détails sur l'erreur rencontrée
      - schema:
          id: Info message
          properties:
            status:
              type: string
              description: code d'info HTTP
            code:
              type: number
              description: code d'info d'application
            info:
              type: string
              description: message court décrivant le type d'info
            message:
              type: string
              description: détails
    responses:
      202:
        description: La création de l'emplacement est passé
        $ref: "#/definitions/Info message"
      400:
        description: La requête est incorecte
        $ref: "#/definitions/Error message"
      401:
        description: Authentification incorects
        $ref: "#/definitions/Error message"
      500:
        description: L'API a recontré une erreur
        $ref: "#/definitions/Error message"
    """
    vacations = VacationRequest(requests_id)
    response = vacations.delete()
    return jsonify(response['content'])


@api.route(
    '/vacations/update/<string:requests_id>',
    methods=['PUT']
)
def update_vacation(requests_id):
    """
    Mettre à jour la demande.
    Ce endpoint permet de modifier \
    les informations d'une demande précise.
    ---
    tags:
      - vacation
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    definitions:
      - schema:
          id: Error message
          properties:
            status:
              type: string
              description: code d'erreur HTTP
            code:
              type: number
              description: code d'erreur d'application
            error:
              type: string
              description: message court décrivant le type d'erreur
            message:
              type: string
              description: détails sur l'erreur rencontrée
      - schema:
          id: Info message
          properties:
            status:
              type: string
              description: code d'info HTTP
            code:
              type: number
              description: code d'info d'application
            info:
              type: string
              description: message court décrivant le type d'info
            message:
              type: string
              description: détails
    responses:
      202:
        description: La création de l'emplacement est passé
        $ref: "#/definitions/Info message"
      400:
        description: La requête est incorecte
        $ref: "#/definitions/Error message"
      401:
        description: Authentification incorects
        $ref: "#/definitions/Error message"
      500:
        description: L'API a recontré une erreur
        $ref: "#/definitions/Error message"
    """
    vacations = VacationRequest(requests_id)
    data = request.get_json()
    date_from = data['date_from']
    date_to = data['date_to']
    name = data['name']
    note = data['note']
    response = vacations.update(date_from, date_to, name, note)
    return jsonify(response['content'])


# EOF
