# -*- coding: utf-8 -*-

import requests
import logging
from datetime import datetime, timedelta

import records
import pytds

from .consts import DB_CREDENTIALS
from .consts import REDIS_DB as re

OPERATORS = {
    "orangemoney": {
        "is_active": True,
        "name": "Orange Money",
        "id_pg": "orange-izytv",
        "code_sage": "ORM",
        "account": "C0000406351",
        "account_gen": "5300035",
        "total": 0,
    },
    "mvola": {
        "is_active": True,
        "name": "Mvola",
        "id_pg": "telma-izytv",
        "code_sage": "MVL",
        "account": "C0000406350",
        "account_gen": "5200019",
        "total": 0,
    },
    "airtelmoney": {
        "active": False,
        "name": "Airtel Money",
        "id_pg": "airtel-izytv",
        "code_sage": "AIRT",
        "account": "C0000406352",
        "account_gen": "5300033",
        "total": 0,
    },
}

yesterday = datetime.now() - timedelta(days=1)
yesterday_db = yesterday.strftime("%d/%m/%Y")
yesterday_redis = yesterday.strftime("%Y-%m-%d")


class Sage:

    def __init__(self, om, mv):
        b = {
            'true': True,
            'false': False,
        }
        OPERATORS['orangemoney']['is_active'] = b[om]
        OPERATORS['mvola']['is_active'] = b[mv]
        self.response = {
            "payload": {},
            "message": "",
            "status_code": 000
        }

    def startAccounting(self):
        """
        Main function for launching all database records.
        Final response returned by API follows the format:
            response = {
                "payload": {
                    "operator1": {
                        "code": return_code_of_operator_transaction_to_sage,
                        "message": return_msg_of_operator_transaction_to_sage,
                        "redis_deletion": {
                            "code": return_code_of_operator_transactions_deleted_from_redis,
                            "message": return_msg_of_operator_transactions_deleted_from_redis,
                        }
                    },
                    "operator2": {
                        ...
                    },
                },
                "message": return_message_of_api_call,
                "status_code": return_code_of_api_call
            }
        """

        try:
            connection = self.connect_sage()
            if not connection:
                self.response['status_code'] = 500
                self.response['message'] = "Impossible d'établir une connexion avec Sage"
                return self.response

            for operator in OPERATORS:
                if operator['is_active']:
                    self.response['payload'][operator] = {}
                    total = self.sum_transactions(operator)
                    self.write_sage(total, operator, connection)
                    self._erase_redis(operator)
        except Exception as e:
            self.response['status_code'] = 500
            self.response['message'] = str(e)
        return self.response

    def database_url(self):
        """Builds the database url. Duplicate from core.py"""

        url = "{eng}://{user}:{password}@{host}:{port}/{name}".format(
            eng=DB_CREDENTIALS["engine"],
            user=DB_CREDENTIALS["user"],
            password=DB_CREDENTIALS["password"],
            host=DB_CREDENTIALS["host"],
            port=DB_CREDENTIALS["port"],
            name=DB_CREDENTIALS["name"],
        )
        # url = "{eng}://{user}:{password}@{host}:{port}/{name}".format(
        #     eng="postgresql",
        #     user="easytv_user",
        #     password="be7Boogh5Joobee",
        #     host="bldb002p.malagasy.com",
        #     port=7432,
        #     name="easytvdb",
        # )
        return url

    def sum_transactions(self, operator):
        """Sums all transactions from the given operator in the postgres database"""

        url = self.database_url(**DB_CREDENTIALS)
        query = (
            "SELECT SUM(montant) FROM izytv_transaction"
            "WHERE datetime_req=={date}"
            "AND partner_id=={id_pg}"
        ).format(date=yesterday_db, operator=OPERATORS[operator]['id_pg'])
        with records.Database(url) as db:
            total = db.query(query)
            return total

    def get_token(self):
        """Used to generate Blueline token for api gateway authentification"""

        token = requests.post('https://api.blueline.mg/dev/auth/request-token', auth=('bpm', '12345678'))
        token = token.json()['token']
        return token

    def connect_sage(self):
        """Connects to sage database using pytds (Python DBAPI driver for MSSQL)"""

        try:
            connection = pytds.connect(
                server="192.168.2.60",
                user="sa",
                password="ANY1793sage",
                database="BLLP_2019_20191204",
            )
            logging.info(connection)
            logging.info("\n\nConnection with SAGE established.\n\n")
            return connection
        except Exception as e:
            logging.error("Error trying to connect with SAGE.")
            # exc_info = sys.exc_info()
            # exception = ''.join(traceback.format_exception(*exc_info))
            logging.error(str(e))
            return None

    def generate_ref(self, operator):
        """Generates a unique reference for a given operator transaction.
        For future reference, note that sage ref cannot exceed 13 chars.
        """
        return "{}{}".format(OPERATORS[operator]['code_sage'], yesterday_db)

    def write_sage(self, total, operator, connection):
        """Builds the sql query and inserts it in sage database."""

        response = self.response['payload'][operator]
        query = ((
            "execute dbo.usp_ins_ec_divers "
            "@jo_num='{code_j}',"
            "@ec_date='{yesterday}',"
            "@ec_echeance='{yesterday}',"
            "@ct_num='{partner_account}',"
            "@ec_intitule='Izytv/encst {operator} du {yesterday}',"
            "@ec_mt={amount},"
            "@cg_credit='4110001',"
            "@cg_debit='{account_gen}',"
            "@ec_piece='{ref}',"
            "@ec_refpiece='{ref}',"
            "@ec_sens='1'"
        ).format(
            operator=operator,
            yesterday=yesterday_db,
            amount=total,
            code_j=OPERATORS[operator]['code_sage'],
            ref=self.generate_ref(operator),
            partner_account=OPERATORS[operator]['account'],
            account_gen=OPERATORS[operator]['account_gen'],
        ))
        with connection.cursor() as cur:
            try:
                cur.execute(query)
                connection.commit()
                response['code'] = 200
                response['message'] = "Transactions insérées avec succès"
                logging.info("\n\nInsertion successful\n\n")
            except Exception as e:
                logging.error(str(e))
                response['code'] = 500
                response['message'] = str(e)

    def _erase_redis(self, operator):
        """Erases transaction in Redis from the given operator, dated yesterday"""

        response = self.response['payload'][operator]['redis_deletion']
        logging.warning('\n\nPreparing to erase data in Redis for operator "{}"...\n\n'.format(operator))
        operator_id = OPERATORS[operator]['id_pg']
        try:
            keys = re.keys()
            logging.warning("\n\nErasing data inside Redis...\n\n")
            for key in keys:
                operator = re.hget(key, 'operator')
                trans_date = re.hget(key, 'TransactionDate')
                if (operator == operator_id) and (yesterday_redis in trans_date):
                    re.delete(key)
                logging.info(
                    "\n\nDeleted {} {} {}\n\n".format(
                        key, operator_id, yesterday_redis
                    )
                )
            logging.info("\n\nRedis data deletion complete.\n\n")
            response['code'] = 500
            response['message'] = "Transactions supprimées dans Redis"
        except Exception as e:
            logging.error(
                '\n\nData deletion for operator "{}" in Redis aborted.\n\n'.format(operator)
            )
            logging.error(str(e))
            response['code'] = 500
            response['message'] = str(e)
