#!/usr/bin/env python
# encoding: utf-8

from datetime import datetime
import requests
import logging
from .consts import CONFIG, LOGS, ORANGE_MONEY, REDIS_DB, IzyTV


class OrangeMoney(object):

    def __init__(self):
        self.url_api = ORANGE_MONEY["url_api"]

    def requestToken(self):
        try:
            response = requests.post(
                ORANGE_MONEY["url_token"],
                headers={
                    'Authorization': 'Basic {}'.format(
                        CONFIG["orangemoney"]["hashcode"]
                    )
                },
                data={'grant_type': 'client_credentials'}
            )
            if response.status_code == 200:
                return response.json()['access_token']
            return None
        except Exception as e:
            LOGS.logger.error("[OrangeMoney] token request: {}".format(e))
            return None

    def paymentRequest(self, data):
        token = self.requestToken()
        if token:
            or_data = {
                "order_id": data["operation_id"],
                "amount": data["amount"],
                "return_url": IzyTV["return_url"],
                "cancel_url": IzyTV["cancel_url"],
                "notif_url": IzyTV["notif_url"],
                "lang": "fr",
                "reference": "MerchantWP00014",
                "merchant_key": ORANGE_MONEY['merchant_key'],
                "currency": ORANGE_MONEY['currency'],
            }
            try:
                response = requests.post(
                    "{}/webpayment".format(ORANGE_MONEY["url_api"]),
                    headers={
                        'content-type': 'application/json',
                        'authorization': 'Bearer {}'.format(token)
                    },
                    json=or_data
                )
                LOGS.logger.info(
                    "[OrangeMoney][{}]{}".format(
                        data["operation_id"],
                        response.json()
                    )
                )

            except Exception as e:
                response = {
                    "code": "046-06-404",
                    "response": {},
                    "message": str(e),
                    "status": 404
                }
                LOGS.logger.error(
                    "[OrangeMoney][{}]{}".format(
                        data["operation_id"],
                        str(e)
                    )
                )

            if response.status_code == 201:
                # Orange always returns 201 regardless of the actual transaction status
                jresponse = response.json()
                if jresponse.get("status"):
                    if jresponse["status"] == 201 and jresponse["message"] == "OK":
                        # save transaction informations into Redis
                        self.writeToRedis(data, jresponse)
                        response = {
                            "code": "046-06-200",
                            "response": {
                                "message": jresponse["message"],
                                "pay_token": jresponse["pay_token"],
                                "notif_token": jresponse["notif_token"],
                                "payment_url": jresponse["payment_url"]
                            },
                            "message": jresponse["message"],
                            "status": 200
                        }
                    else:
                        response = {
                            "code": "046-06-{}".format(jresponse["status"]),
                            "response": {
                                "message": jresponse["message"],
                                "pay_token": jresponse["pay_token"],
                                "notif_token": jresponse["notif_token"],
                                "payment_url": jresponse["payment_url"]
                            },
                            "message": jresponse["message"],
                            "status": jresponse["status"]
                        }
                elif jresponse.get("code"):
                    response = {
                        "code": "046-06-{}".format(jresponse["code"]),
                        "response": {
                            "message": jresponse["description"],
                            "pay_token": None,
                            "notif_token": None,
                            "payment_url": None
                        },
                        "message": jresponse["message"],
                        "status": jresponse["code"]
                    }
            else:  # Orange couldn't process the transaction request
                response = {
                    "code": "046-06-{}".format(response.status_code),
                    "response": {},
                    "message": "Serveur Orange Money introuvable",
                    "status": response.status_code
                }
                LOGS.logger.error(
                    "[OrangMoney][{}]{}".format(
                        data["operation_id"],
                        response,
                    )
                )
        return response

    def writeToRedis(self, data, jresponse):
        """Writes the initial transaction inside Redis for asynchronous status checks. """
        today = datetime.today().strftime("%Y-%m-%d %H:%M:%S")
        re_data = {
            "TransactionDate": today,
            "amount": data["amount"],
            "order_id": data["operation_id"],
            "label": data["label"],
            "device_id": data["device_id"],
            "other": data["other"],
            "client_refnum": data["client_refnum"],
            "offre_refnum": data["offre_refnum"],
            "partner_ref": jresponse["pay_token"],
            "operator": "orange-izytv",
            "caller_num": "320000000",
            "service_type": "izytv",
            "status": "En attente de paiement",
            "info": "Activation en attente",
        }
        LOGS.logger.info(
            "[OrangeMoney][{}][write in Redis] {}\n".format(
                data["operation_id"],
                re_data,
            )
        )
        redis_key = "request:payment:{}".format(jresponse["pay_token"])
        REDIS_DB.hmset(redis_key, re_data)

# transaction status request related functions

    def checkTransactionStatus(self, transaction):
        """
        Main function for checking the status of a transaction at Orange.
        """
        order_id = transaction["order_id"]
        redis_key = "request:payment:{}".format(transaction["pay_token"])
        auth_token = self.requestToken()
        if auth_token:
            query = {
                "order_id": transaction["order_id"],
                "amount": transaction["amount"],
                "pay_token": transaction["pay_token"],
            }
            try:
                response = requests.post(
                    "{}/transactionstatus".format(ORANGE_MONEY["url_api"]),
                    headers={
                        'content-type': 'application/json',
                        'authorization': 'Bearer {}'.format(auth_token)
                    },
                    json=query
                )
                LOGS.logger.info(
                    "[OrangeMoney][{}]{}\n".format(
                        order_id,
                        response.json(),
                    )
                )
            except Exception as e:
                LOGS.logger.error(
                    "[OrangeMoney][{}]{}\n".format(
                        order_id,
                        str(e),
                    )
                )
                response = {
                    "code": "046-06-404",
                    "response": {},
                    "message": str(e),
                    "status": 404
                }
                return response

            if response.status_code == 201:
                # Orange returns 201 regardless of the actual transaction status
                jresponse = response.json()
                response = {
                    "code": "046-06-000",
                    "response": {
                        "ResponseCode": jresponse['status'],
                        "order_id": jresponse['order_id'],
                        "txnid": jresponse['txnid'],
                    },
                    "message": jresponse['status'],
                    "status": jresponse['status'],
                }
                if jresponse['status'] == 'SUCCESS':
                    response["code"] = "046-06-200"
                    self.update_Redis(
                        redis_key,
                        {
                            "status": "Transaction validée",
                            "partner_ref": jresponse["txnid"],
                        }
                    )
                elif jresponse['status'] in ['EXPIRED', 'FAILED', 'NOT FOUND']:
                    response["code"] = "046-06-000"
                    self.update_Redis(
                        redis_key,
                        {
                            "status": "Transaction annulée",
                        }
                    )
                elif jresponse['status'] in ['PENDING', 'INITIATED']:
                    response["code"] = "046-06-000"
                LOGS.logger.info(
                    "[OrangeMoney][] {}\n".format(
                        order_id,
                        response,
                    )
                )
            return response

    def update_Redis(self, redis_key, dict):
        """
        Updates the required fields in Redis from the given dictionnary.
        """
        REDIS_DB.hmset(
            redis_key,
            dict,
        )

    def updateRefPartner(self, transaction):
        """
        For some unknown reasons, we can update directly a key-value dictionnary pair
        from the shell in Redis, but when it goes through our service, it doesn't
        work. So we need to pass again ALL the dictionnary inside Redis to
        overwrite the entry...
        """
        redis_key = transaction.get("redis_key")
        old_entry = REDIS_DB.hgetall(redis_key)
        new_entry = {
            "TransactionDate": old_entry[b"TransactionDate"],
            "amount": old_entry[b"amount"],
            "order_id": old_entry[b"order_id"],
            "label": old_entry[b"label"],
            "device_id": old_entry[b"device_id"],
            "other": old_entry[b"other"],
            "client_refnum": old_entry[b"client_refnum"],
            "offre_refnum": old_entry[b"offre_refnum"],
            "caller_num": old_entry[b"caller_num"],
            "status": old_entry[b"status"],
            "operator": old_entry[b"operator"],
            "service_type": old_entry[b"service_type"],
            "info": old_entry[b"info"],
            "partner_ref": transaction.get('tnxid'),
        }
        REDIS_DB.delete(redis_key)
        REDIS_DB.hmset(
            redis_key,
            new_entry,
        )
        current_record = REDIS_DB.hgetall(redis_key)
        LOGS.logger.info(
            "[OrangeMoney] {}\n".format(current_record)
        )

# status requests functions

    def checkTransactionStatusRedis(self, order_id):
        """
        Used by 4D in order to check a given transaction current status.
        Value inside Redis is returned.
        """

        redis_key = "request:payment:{}".format(order_id)
        val = REDIS_DB.hgetall(redis_key.encode('utf-8'))
        data = {key.decode('utf-8'): value.decode('utf-8') for (key, value) in val.items()}
        response = {
            "code": 200,
            "response": {
                "order_id": data["partner_ref"]
            },
            "message": data["info"],
            "status": data["status"]
        }
        return response

# EOF
