#!/usr/bin/env python
# encoding: utf-8


from flask import request, jsonify

from blueflask.lib.decorators import auth as ldap
from blueflask.lib.infos import accepted, HTTPInfo
from blueflask.lib.errors import bad_request, HTTPError

from ...consts import SERVICE_CODE
from ...core import Sms
from . import api


@api.route('/sms', methods=['POST'])
@ldap.login_required
def send_sms():
    """
    Envoi de SMS.
    Ce endpoint permet d'envoyer un SMS à un numéro précis.
    ---
    tags:
      - sms
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    definitions:
      - schema:
          id: Message d'erreur
          properties:
            status:
              type: string
              description: code d'erreur HTTP
            code:
              type: number
              description: code d'erreur d'application
            error:
              type: string
              description: message court décrivant le type d'erreur
            message:
              type: string
              description: détails sur l'erreur rencontrée
      - schema:
          id: Message d'information
          properties:
            status:
              type: string
              description: code d'info HTTP
            code:
              type: number
              description: code d'info d'application
            info:
              type: string
              description: message court décrivant le type d'info
            message:
              type: string
              description: détails
    responses:
      202:
        description: Le SIMBank a accepté la requête
        $ref: "#/definitions/Message d'information"
      400:
        description: La requête est incorrecte
        $ref: "#/definitions/Message d'erreur"
      401:
        description: Authentification incorrecte
        $ref: "#/definitions/Message d'erreur"
      500:
        description: L'API a renconté une erreur
        $ref: "#/definitions/Message d'erreur"
    """
    data = request.get_json()
    if not data or 'message' not in data or 'destination' not in data:
        return bad_request(
            "veuillez fournir un document JSON valide",
            service_code=SERVICE_CODE
            )
    else:
        short_message = Sms(
            data['source'],
            data['destination'],
            data['message']
            )
        response = short_message.send(headers=request.headers)
        status = response['status']
        content = response['content']
        headers = response['headers']
        if isinstance(content, dict):
            response = jsonify(content)
            response.status_code = status
            response.headers = headers
        elif status >= 400:
            return HTTPError(status).response(message=content)
        elif status == 200:
            return accepted(service_code=SERVICE_CODE)
        else:
            return HTTPInfo(status).response(message=content)

# EOF
