# -*- coding: utf-8 -*-

from flask import request

from blueflask.lib.infos import success
from blueflask.lib.errors import internal_error

from ...consts import SERVICE_CODE
from . import api

import json

import pika


@api.route('/sms', methods=['POST'])
def send_sms():
    """
    Envoi de SMS.
    Ce endpoint permet d'envoyer un SMS à un numéro précis.
    ---
    tags:
      - sms
    components:
      securitySchemes:
        basicAuth:
          type: http
          scheme: basic
    security:
      - basicAuth: []
    definitions:
      - schema:
          id: Message d'erreur
          properties:
            status:
              type: string
              description: code d'erreur HTTP
            code:
              type: number
              description: code d'erreur d'application
            error:
              type: string
              description: message court décrivant le type d'erreur
            message:
              type: string
              description: détails sur l'erreur rencontrée
      - schema:
          id: Message d'information
          properties:
            status:
              type: string
              description: code d'info HTTP
            code:
              type: number
              description: code d'info d'application
            info:
              type: string
              description: message court décrivant le type d'info
            message:
              type: string
              description: détails
    responses:
      202:
        description: Le SIMBank a accepté la requête
        $ref: "#/definitions/Message d'information"
      400:
        description: La requête est incorrecte
        $ref: "#/definitions/Message d'erreur"
      401:
        description: Authentification incorrecte
        $ref: "#/definitions/Message d'erreur"
      500:
        description: L'API a renconté une erreur
        $ref: "#/definitions/Message d'erreur"
    """
    auth = request.authorization
    if auth:
        auth = (auth.username, auth.password)
    params = request.args or {}
    headers = request.headers or {}
    json_doc = request.get_json() or {}
    sender_id = json_doc.get('sender_id', 'default')
    document = {
        'auth': auth or (),
        'params': params,
        'headers': headers,
        'json': json_doc
        }
    try:
        connection = pika.BlockingConnection(
            pika.ConnectionParameters(
                host='localhost',
                virtual_host='/simbank'
                )
            )
    except Exception:
        # exception de connection lancée par pika
        # il faut que rabbitmq-server soit installé
        # et que le vhost /{service} existe déjà
        return internal_error(
            message="serveur RabbitMQ injoignable",
            code='{}-5000'.format(SERVICE_CODE),
            service_code=SERVICE_CODE
            )
    else:
        channel = connection.channel()
        channel.exchange_declare(
            exchange='sendsms',
            exchange_type='topic'
            )
        # déclaration idempotente: pas grave si l'exchange existe déjà
        channel.basic_publish(
            exchange='sendsms',
            routing_key=sender_id,
            body=json.dumps(document),
            properties=pika.BasicProperties(delivery_mode=2)
            )
        connection.close()
        return success(service_code=SERVICE_CODE)

# EOF
