# -*- coding: utf-8 -*-

from spyne import rpc, ServiceBase, Unicode

from ...gateway import BadRequestError, SoapPayload

import xmltodict


class FoamService(ServiceBase):

    @rpc(
        Unicode,
        _returns=Unicode,
        _operation_name='BubbleRequest',
        _out_message_name='BubbleResponse',
        _in_variable_names={'payload': 'xml_in'},
        _out_variable_name='xml_out')
    def bubble(ctx, payload):
        """
        Soap service that will translate the string you passed
        into a valid REST request.

        Let's say you want to perform the following request in REST:

        GET --auth a_token: /service/v2/some_ressource/{p1}/{p2}?a=x&b=y

        With this service (BubbleRequest), you would send something like this:

        <ns:BubbleRequest xmlns:ns="api.blueline.mg">
          <ns:xml_in>
            <?xml version="1.0" encoding="utf-8"?>
            <root>
              <header>
                <version>2</version>
                <param1>GET</param1>
                <param2>service</param2>
                <param3>/some_ressource</param3>
                <ident></ident>
                <psw></psw>
              </header>
              <data>
                <token>a_token</token>
                <resource_params>
                  <param1>p1</param1>
                  <param2>p2</param2>
                </resource_params>
                <query_params>
                  <a>x</a>
                  <b>y</b>
                </query_params>
                <json_params>
                  <key1>value1</key1>
                  <key2>value2</key2>
                </json_params>
              </data>
            </root>
          </ns:xml_in>
        </ns:BubbleRequest xmlns:ns="api.blueline.mg">
        """

        try:
            source = ctx.transport.req['HTTP_X_REAL_IP']
            payload = SoapPayload(payload, source=source)  # sick redundancy
        except SyntaxError:
            # respond with a Fault
            raise BadRequestError()
        else:
            response = payload.translate()
            response = xmltodict.unparse(response, pretty=True)
            response = '\n'.join(response.split('\n')[1:])
            return response

# EOF
