#!/usr/bin/env python
# encoding: utf-8
import json
import rest_client
import pika

from flask import jsonify
from blueflask.lib.errors import bad_request
from . import CONFIG
from . import LOGS
from . import URL_STATUS
from . import URL_STATUS_AIRTEL
from .core import check_parameters
from .core import transaction_date

SERVICE_CODE = "040-05"


class OCS(object):
    """
    une classe pour les requêtes à destination de l'api OCS
    """

    def __init__(self):
        self.url = CONFIG["SERVICES"]["ocs"]
        self.client = rest_client.RestClient(self.url)

    def authentication(self):
        """
        Cette méthode permettra au partenaire d'avoir
        la liste des forfaits possible pour les
        offres bip
        A l'avenir on pourra l'exploiter pour autre chose
        """
        forfait_list = CONFIG["MISC"]["bip_offer_type"]
        response = {
            "code": "040-05-200",
            "info": "authentification menu bip",
            "message": {"forfait_list": forfait_list},
            "status": 200,
        }
        return jsonify(response)

    def get_language(self, **data):
        LOGS.logger.info(data)
        data_headers = data["headers"]
        token = data["token"]
        caller_num = data["caller_num"]
        response = self.client.get(
            "/subscribers/{caller_num}".format(caller_num=caller_num),
            auth=(token, ""),
            headers=data_headers,
            params={"key": "language_code"},
        )
        return {"status": response["status"], "data": response["content"]["message"]}

    @check_parameters("caller_num", "language_code")
    def set_language(self, token, **data):
        request = data["request"]
        caller_num = data["caller_num"]
        del data["request"]
        response = self.client.put(
            "/subscribers/{caller_num}".format(caller_num=caller_num),
            auth=(token, ""),
            headers=request.headers,
            json={"language_code": data["language_code"]},
        )
        return {"status": response["status"], "data": response["content"]["message"]}

    @check_parameters("caller_num")
    def has_code(self, token, **data):
        request = data["request"]
        del data["request"]
        response = self.client.get(
            "/auth/{}".format(data["caller_num"]),
            auth=(token, ""),
            headers=request.headers,
        )
        return {"status": response["status"], "data": response["content"]}

    @check_parameters("caller_num", "secret_code")
    def verify_code(self, token, **data):
        request = data["request"]
        del data["request"]
        response = self.client.post(
            "/auth/verify-code",
            auth=(token, ""),
            headers=request.headers,
            json={"phonenumber": data["caller_num"], "code": data["secret_code"]},
        )
        return {"status": response["status"], "data": response["content"]}

    @check_parameters("caller_num", "secret_code")
    def set_code(self, token, **data):
        request = data["request"]
        del data["request"]
        response = self.client.post(
            "/auth",
            auth=(token, ""),
            headers=request.headers,
            json={"phonenumber": data["caller_num"], "code": data["secret_code"]},
        )
        return {"status": response["status"], "data": response["content"]}

    @check_parameters("customer_id")
    def get_infos(self, token, **data):
        """
        On reçoit de l'API OCS un document comme ceci:
        {
            'phonenumber': 0390390169,
            'length': 2;
            'infos': [
                {
                'name': 'principal',
                'balance': '200',
                },
                {
                'name': 'Tsik machin',
                'balance': '0s, 40MO, 4sms',
                },
            ]
        }
        """
        request = data["request"]
        phonenumber = data["customer_id"]
        del data["customer_id"]
        del data["request"]
        response = self.client.get(
            "/subscribers/{phonenumber}".format(phonenumber=phonenumber),
            auth=(token, ""),
            headers=request.headers,
        )
        if response["status"] == 200:
            infos = [
                {"name": info["name"], "value": info["balance"]}
                for info in response["content"]["message"]["subscriptions"]
            ]
            result = {"status": response["status"], "data": infos}
            return result
        elif response["status"] == 204:
            infos = []
            result = {"status": response["status"], "data": infos}
            return result
        else:
            return {
                "status": response["status"],
                "data": response["content"]["message"],
            }

    @check_parameters("customer_id", "balance")
    def get_offers(self, token, **data):
        """
        Ceci est un endpoint qui noue permettra de récupérer
        la liste des offres bip disponible
        On utilisera le paramère "type" pour filtrer
        selon le type de forfait qu'on veut afficher
        sinon il affichera tout
        """
        request = data["request"]
        LOGS.logger.info("Le header c'est par ici: {}".format(request.headers))
        response = self.client.get("/services/offers", auth=(token, ""))
        if response["status"] == 200:
            if data["service_type"] == "bip":
                infos = [offer for offer in response["content"]["message"]["offers"]]
                if "type" in data:
                    infos = []
                    forfait = data["type"]
                    LOGS.logger.info(forfait)
                    for offer in response["content"]["message"]["offers"]:
                        if offer["tag"] == forfait:
                            infos.append(offer)
            else:
                LOGS.logger.info("Vorondolo...voir a quoi ce code sert-il?")
                infos = [
                    offer
                    for offer in response["content"]["message"]["offers"]
                    if float(offer["price"]) <= float(data["balance"])
                ]
        else:
            infos = response["content"]["message"]
        result = {"status": response["status"], "data": infos}
        return result

    @check_parameters("offer_id", "customer_id")
    def buy_voucher(self, token, **data):
        phonenumber = data["customer_id"]
        # buy voucher
        request = data["request"]
        del data["request"]
        code = data["offer_id"]
        response = self.client.post(
            "/services/vouchers",
            auth=(token, ""),
            headers=request.headers,
            json={"phonenumber": phonenumber, "code": code},
        )
        return {"status": response["status"], "data": response["content"]["message"]}

    @check_parameters("offer_id", "customer_id")
    def buy_offer(self, token, **data):
        """
        methode à utiliser pour l'achat
        d'une offre bip
        """
        phonenumber = data["customer_id"]
        request = data["request"]
        del data["request"]
        offer_id = data["offer_id"]
        response = self.client.post(
            "/services/offers",
            auth=(token, ""),
            headers=request.headers,
            json={"phonenumber": phonenumber, "offer_id": offer_id},
        )
        return {"status": response["status"], "data": response["content"]["message"]}

    def write_to_db(**data):
        pass

    @check_parameters("customer_id", "device_name", "amount")
    def transfer(self, token, **data):
        """
        Transfert de credit entre abonnés bip
        """
        request = data["request"]
        del data["request"]
        receiver = data["device_name"]
        sender = data["customer_id"]
        amount = data["amount"]
        infos = "transfert de credit venant de: {}".format(data["customer_id"])
        response = self.client.post(
            "/services/transfers",
            auth=(token, ""),
            headers=request.headers,
            json={
                "sender": sender,
                "receiver": receiver,
                "amount": amount,
                "infos": infos,
            },
        )
        return {"status": response["status"], "data": response["content"]["message"]}

    @check_parameters("customer_id")
    def check_balance(self, token, **data):
        """
        Récupération du solde principal téléphonique (2000) d'un abonné
        """
        request = data["request"]
        phonenumber = data["caller_num"]
        del data["service_type"]
        del data["customer_id"]
        del data["request"]
        response = self.client.get(
            "/subscribers/{phonenumber}/balances".format(phonenumber=phonenumber),
            auth=(token, ""),
            headers=request.headers,
            json={"account_type": "2000"},
        )
        return {"status": response["status"], "data": response["content"]["message"]}

    @check_parameters("customer_id", "amount")
    def update_balance(self, token, **data):
        """
        Modification du solde téléphonique (2000) d'un abonné
        bip
        """
        response = self.client.put(
            "/subscribers/{phonenumber}/balances".format(
                phonenumber=data["caller_num"]
            ),
            auth=(token, ""),
            headers=data["request"].headers,
            json={
                "amount": data["amount"],
                "operation_type": "2",
                "account_type": 2000,
            },
        )
        return {"status": response["status"], "data": response["content"]["message"]}

    def process(self, message):
        """
        ceci est pour le traitement en asynchrone des achats
        d'offres bip ou pour le transfert de sou
        mobile_money_to_ocs
        """
        connection = pika.BlockingConnection(
            pika.ConnectionParameters(host="localhost", virtual_host="/ussd")
        )
        channel = connection.channel()
        channel.exchange_declare(exchange="ussd_offer", exchange_type="topic")
        channel.basic_publish(
            routing_key="ussd_msg",
            exchange="ussd_offer",
            properties=pika.BasicProperties(
                app_id="ussd-services", delivery_mode=2  # rendre le message persistant
            ),
            body=message,
        )
        connection.close()

    @check_parameters("customer_id", "amount", "caller_num")
    def transfer_credit_partner(self, token, **data):
        """
        ceci permettra au partenaire d'acheter un credit
        bip depuis leur mobile money vers un compte bip
        le traitement ce fait d'une manière asynchrone
        """
        request = data["request"]
        headers = request.headers
        request_id = headers["X-Request-Id"]
        activation_data = {
            "amount": data["amount"],
            "operator": data["operator"],
            "service_type": data["service_type"],
            "caller_num": data["caller_num"],
            "partner_ref": data["partner_ref"],
            "device_id": data["device_name"],
            "offer_name": data["offer_name"],
            "request_id": request_id,
            "customer_id": data["customer_id"],
            "headers": headers,
            "token": token,
            "transaction_date": transaction_date(),
        }
        result = {
            "data": activation_data,
            "status": 200,
            "info": "ack",
            "message": "traitement en cours",
        }
        LOGS.logger.info(result)
        message = json.dumps(result)
        self.process(message)
        # definir l'url statut selon l'operateur
        if data["operator"] == "airtel":
            url_status = URL_STATUS_AIRTEL
        else:
            url_status = URL_STATUS
        details = "{}/ussd/v1/clients/{}/{}".format(
            url_status, data["operator"], request_id
        )
        # refaire un code review peut etre
        response = {
            "status": result["status"],
            "info": "traitement en cours",
            "message": details,
            "code": "040-05-200",
        }
        return response

    @check_parameters("offer_id", "customer_id")
    def buy_offer_partner(self, token, **data):
        """
        methode à utiliser pour l'achat
        d'une offre bip pour le cas d'un partenaire
        l'achat se fait en mode asyncrhone :D
        """
        request = data["request"]
        headers = request.headers
        request_id = headers["X-Request-Id"]
        try:
            activation_data = {
                "amount": data["amount"],
                "operator": data["operator"],
                "service_type": data["service_type"],
                "caller_num": data["caller_num"],
                "partner_ref": data["partner_ref"],
                "device_id": data["device_name"],
                "bundle_name": data["bundle_name"],
                "offer_name": data["offer_name"],
                "offer_id": data["offer_id"],
                "request_id": request_id,
                "customer_id": data["customer_id"],
                "headers": headers,
                "token": token,
                "transaction_date": transaction_date(),
            }
            result = {
                "data": activation_data,
                "status": 200,
                "info": "ack",
                "message": "traitement en cours",
            }
            message = json.dumps(result)
            LOGS.logger.info(message)
            self.process(message)
            if data["operator"] == "airtel":
                url_status = URL_STATUS_AIRTEL
            else:
                url_status = URL_STATUS
            details = "{}/ussd/v1/clients/{}/{}".format(
                url_status, data["operator"], request_id
            )
            response = {
                "status": result["status"],
                "info": "traitement en cours",
                "message": details,
                "code": "040-05-200",
            }
            LOGS.logger.info(
                "Demande d'achat d'une offre bip reussie: {}".format(result)
            )
            return response
        except SyntaxError:
            LOGS.logger.info(
                "Demande d'achat d'une offre bip echouee: {}".format(result)
            )
            return bad_request(message="paramètres incomplets", code="040-05-400")
